<?php

namespace Drupal\images_import\Services;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for removing a image.
 */
class RemoveImage {
  use StringTranslationTrait;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $logger;

  /**
   * Constructs a ProcessSingleImage object.
   */
  public function __construct(
    FileSystemInterface $file_system,
    EntityTypeManagerInterface $entity_type_manager,
    FileRepositoryInterface $file_repository,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->fileSystem = $file_system;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileRepository = $file_repository;
    $this->messenger = $messenger;
    $this->logger = $logger_factory;
  }

  /**
   * Remove a single image from the entity.
   */
  public function removeImage($row, $images_options) {
    $entity_type = $images_options['entity_type'];
    $image_field = $images_options['image_field'];

    // Check if entity_id exist then Load based on that.
    if (isset($row['entity_id']) && !empty($row['entity_id'])) {
      $entity = $this->entityTypeManager->getStorage($entity_type)->load($row['entity_id']);
      if (!$entity) {
        $this->messenger->addError($this->t('Entity with ID @id not found.', [
          '@id' => $row['entity_id'],
        ]));
        return FALSE;
      }
    }
    elseif (isset($row['entity_title'])) {
      // Load entity by title and remove image field value.
      if ($entity_type === 'user') {
        $entities = $this->entityTypeManager->getStorage('user')->loadByProperties([
          'name' => $row['entity_title'],
        ]);
      }
      elseif ($entity_type === 'block_content') {
        $entities = $this->entityTypeManager->getStorage('block_content')->loadByProperties([
          'info' => $row['entity_title'],
        ]);
      }
      elseif ($entity_type === 'taxonomy_term') {
        $entities = $this->entityTypeManager->getStorage('taxonomy_term')->loadByProperties([
          'name' => $row['entity_title'],
        ]);
      }
      else {
        $entities = $this->entityTypeManager->getStorage($entity_type)->loadByProperties([
          'title' => $row['entity_title'],
        ]);
      }
      if (empty($entities)) {
        $this->messenger->addError($this->t('Entity with title @title not found.', [
          '@title' => $row['entity_title'],
        ]));
        return FALSE;
      }
      // Get the first matched entity.
      $entity = reset($entities);
    }
    else {
      $this->messenger->addError($this->t('No entity_id or entity_title provided in the CSV row.'));
      return FALSE;
    }

    // Remove image field value.
    if ($entity->hasField($image_field) && !$entity->get($image_field)->isEmpty()) {
      $entity->set($image_field, NULL);
      $entity->save();
      $this->logger->get('images_import')->info($this->t('Removed image/s from @entity_type @entity_title', [
        '@entity_type' => $entity_type,
        '@entity_title' => $entity_type === 'user' ? $entity->getAccountName() : $entity->getTitle(),
      ]));
      return TRUE;
    }
    else {
      $this->messenger->addError($this->t('Entity @entity_type @entity_title does not have an image/s in field @field or field is empty.', [
        '@entity_type' => $entity_type,
        '@entity_title' => $entity_type === 'user' ? $entity->getAccountName() : $entity->getTitle(),
        '@field' => $image_field,
      ]));
      return FALSE;
    }
  }

}
