<?php

namespace Drupal\images_import\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\images_import\Services\BatchProcessor;

/**
 * Provides a form for image removal via CSV.
 */
class RemoveExistingImageForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new RemoveExistingImageForm.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    FileSystemInterface $file_system,
    FileRepositoryInterface $file_repository,
    MessengerInterface $messenger,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->fileSystem = $file_system;
    $this->fileRepository = $file_repository;
    $this->messenger = $messenger;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('file_system'),
      $container->get('file.repository'),
      $container->get('messenger'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'remove_existing_images_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get all entity types that have fields.
    $entity_types = $this->entityTypeManager->getDefinitions();
    $entity_type_options = [];

    foreach ($entity_types as $entity_type_id => $entity_type) {
      if ($entity_type->get('field_ui_base_route')) {
        $entity_type_options[$entity_type_id] = $entity_type->getLabel();
      }
    }

    // Limit to common content entities for simplicity.
    $commerce_product_enabled = $this->moduleHandler->moduleExists('commerce_product');
    $common_entities = [
      "block_content" => 'Content block',
      "node" => 'Content',
      $commerce_product_enabled ? "commerce_product" : NULL => 'Product',
      "taxonomy_term" => 'Taxonomy term',
    ];

    $csv_file = $this->config('remove_existing_images_form.settings')->get('csv_file');
    $fid = $csv_file ? reset($csv_file) : NULL;

    // file_validate_extensions is deprecated in drupal:10.2.0
    // and is removed from drupal:11.0.0.
    $validator_key = version_compare(\Drupal::VERSION, '10.2', '>=') ? 'FileExtension' : 'file_validate_extensions';

    $form['csv_file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('CSV File'),
      '#description' => $this->t('Upload a CSV file with entity_title columns. <a href="/admin/config/content/image-import/demo" download="demo-image-import.csv">Download demo CSV</a>'),
      '#upload_validators' => [
        $validator_key => ['csv'],
      ],
      '#upload_location' => 'public://image_import/',
      '#required' => TRUE,
      '#default_value' => $fid ? [$fid] : NULL,
    ];

    $selected_entity_type = $form_state->getValue('entity_type');

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity Type'),
      '#options' => $common_entities,
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateBundles',
        'wrapper' => 'bundle-wrapper',
        'event' => 'change',
      ],
    ];

    $bundle_options = $selected_entity_type ? $this->getBundles($selected_entity_type) : [];
    $form['bundle'] = [
      '#type' => 'select',
      '#title' => $this->t('Bundle'),
      '#options' => $bundle_options,
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#prefix' => '<div id="bundle-wrapper">',
      '#suffix' => '</div>',
      '#ajax' => [
        'callback' => '::updateImageFields',
        'wrapper' => 'image-field-wrapper',
        'event' => 'change',
      ],
    ];

    $form['image_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Image Field'),
      '#options' => $this->getImageFields($form_state->getValue('entity_type'), $form_state->getValue('bundle')),
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#prefix' => '<div id="image-field-wrapper">',
      '#suffix' => '</div>',
      '#description' => $this->t('Select the image field to remove images from.'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save and Remove Images'),
    ];

    return $form;
  }

  /**
   * AJAX callback to update bundles based on entity type.
   */
  public function updateBundles(array &$form, FormStateInterface $form_state) {
    return $form['bundle'];
  }

  /**
   * AJAX callback to update image fields based on entity type and bundle.
   */
  public function updateImageFields(array &$form, FormStateInterface $form_state) {
    return $form['image_field'];
  }

  /**
   * Get bundles for the selected entity type.
   */
  protected function getBundles($entity_type_id) {
    if (!$entity_type_id) {
      return [];
    }

    $bundle_options = [];
    if (($entity_type_id == "taxonomy_term")) {
      $vocabularies = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple();
      foreach ($vocabularies as $vocabulary) {
        $bundle_options[$vocabulary->id()] = $vocabulary->label();
      }
    }
    else {
      $bundles = $this->entityTypeManager->getStorage($entity_type_id . '_type')->loadMultiple();
      foreach ($bundles as $bundle) {
        $bundle_options[$bundle->id()] = $bundle->label();
      }
    }

    return $bundle_options;
  }

  /**
   * Get image fields for the selected entity type and bundle.
   */
  protected function getImageFields($entity_type_id, $bundle_id) {
    if (!$entity_type_id or !$bundle_id) {
      return [];
    }

    $field_options = [];
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);

    foreach ($field_definitions as $field_name => $field_definition) {
      if ($field_definition->getType() === 'image') {
        $field_options[$field_name] = $field_definition->getLabel();
      }
      elseif (
        $field_definition->getType() === 'entity_reference' &&
        $field_definition->getSetting('target_type') === 'media'
      ) {
        // Check if this media reference field only allows 'image' media bundle.
        $handler_settings = $field_definition->getSetting('handler_settings');
        if (!empty($handler_settings['target_bundles'])) {
          $media_bundles = $handler_settings['target_bundles'];
          foreach ($media_bundles as $bundle) {
            $media_bundle = $this->entityTypeManager->getStorage('media_type')->load($bundle);
            if ($media_bundle && $media_bundle->get('source') === 'image') {
              $field_options[$field_name] = $field_definition->getLabel();
              break;
            }
          }
        }
      }
    }

    return $field_options;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $csv_file = $form_state->getValue('csv_file');
    if (!empty($csv_file)) {
      $file = $this->entityTypeManager->getStorage('file')->load($csv_file[0]);
      if ($file) {
        $file_path = $this->fileSystem->realpath($file->getFileUri());
        $this->validateCsvFile($file_path, $form_state);
      }
    }
  }

  /**
   * Validate CSV file structure.
   */
  protected function validateCsvFile($file_path, FormStateInterface $form_state) {
    if (($handle = fopen($file_path, 'r')) !== FALSE) {
      $header = fgetcsv($handle);
      fclose($handle);

      $required_columns = ['entity_title'];
      $missing_columns = array_diff($required_columns, $header);

      if (!empty($missing_columns)) {
        $form_state->setErrorByName('csv_file', $this->t('CSV file is missing required columns: @columns', [
          '@columns' => implode(', ', $missing_columns),
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $csv_file_id = $form_state->getValue('csv_file')[0];
    $entity_type = $form_state->getValue('entity_type');
    $bundle = $form_state->getValue('bundle');
    $image_field = $form_state->getValue('image_field');

    $images_options = [
      'entity_type' => $entity_type,
      'bundle' => $bundle,
      'image_field' => $image_field,
      'operation' => 'remove',
    ];

    // Save file as permanent.
    $file = $this->entityTypeManager->getStorage('file')->load($csv_file_id);
    $file->setPermanent();
    $file->save();
    $this->configFactory()->getEditable('remove_existing_images_form.settings')->set('csv_file', $form_state->getValue('csv_file'))
      ->save();

    $file_path = $this->fileSystem->realpath($file->getFileUri());

    if (($handle = fopen($file_path, 'r')) !== FALSE) {
      $header = fgetcsv($handle);

      $rows = [];
      while (($data = fgetcsv($handle)) !== FALSE) {
        $row = array_combine($header, $data);
        $rows[] = $row;
      }
      fclose($handle);

      // Prepare batch operations.
      $batch_builder = (new BatchBuilder())
        ->setTitle($this->t('Removing images...'))
        ->setInitMessage($this->t('Starting image removal...'))
        ->setProgressMessage($this->t('Processed @current out of @total.'))
        ->setErrorMessage($this->t('An error occurred during image import.'))
        ->setFinishCallback([BatchProcessor::class, 'removeFinishedCallback']);

      foreach ($rows as $row) {
        $batch_builder->addOperation(
          [BatchProcessor::class, 'processItem'],
          [$row, $images_options]
        );
      }

      batch_set($batch_builder->toArray());
      return;
    }
    else {
      $this->messenger->addError($this->t('Failed to open CSV file.'));
      return;
    }
  }

}
