<?php

namespace Drupal\images_import\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\images_import\Services\BatchProcessor;

/**
 * Provides a form for image import via CSV.
 */
class UserImageImportForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new ImageImportForm.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    FileSystemInterface $file_system,
    FileRepositoryInterface $file_repository,
    MessengerInterface $messenger,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->fileSystem = $file_system;
    $this->fileRepository = $file_repository;
    $this->messenger = $messenger;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('file_system'),
      $container->get('file.repository'),
      $container->get('messenger'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_image_import_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $csv_file = $this->config('user_image_import_form.settings')->get('csv_file');
    $fid = $csv_file ? reset($csv_file) : NULL;

    // file_validate_extensions is deprecated in drupal:10.2.0
    // and is removed from drupal:11.0.0.
    $validator_key = version_compare(\Drupal::VERSION, '10.2', '>=') ? 'FileExtension' : 'file_validate_extensions';

    $form['csv_file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('CSV File'),
      '#description' => $this->t('Upload a CSV file with entity_title and image_url columns. <a href="/admin/config/content/image-import/demo" download="demo-image-import.csv">Download demo CSV for single image </a> or <a href="/admin/config/content/image-import/demo-multiple-images" download="demo-multiple-image-import.csv">Download demo CSV for multiple images</a>'),
      '#upload_validators' => [
        $validator_key => ['csv'],
      ],
      '#upload_location' => 'public://image_import/',
      '#required' => TRUE,
      '#default_value' => $fid ? [$fid] : NULL,
    ];

    $form['image_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Image Field'),
      '#options' => $this->getImageFields('user'),
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#prefix' => '<div id="image-field-wrapper">',
      '#suffix' => '</div>',
      '#description' => $this->t('Select the image field to import images into.'),
    ];

    $form['is_multiple'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Is Multiple?'),
      '#description' => $this->t('Check this box if the entity has multiple images. This will append images instead of replacing them. And provide comma separated image URLs in CSV.'),
      '#required' => FALSE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save and Import'),
    ];

    return $form;
  }

  /**
   * Get image fields for the selected entity type and bundle.
   */
  protected function getImageFields($entity_type_id) {
    if (!$entity_type_id) {
      return [];
    }

    $field_options = [];
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $entity_type_id);

    foreach ($field_definitions as $field_name => $field_definition) {
      // Only add fields of type 'image',
      // or 'entity_reference' that reference 'media' entities of type 'image'.
      if ($field_definition->getType() === 'image') {
        $field_options[$field_name] = $field_definition->getLabel();
      }
      elseif (
        $field_definition->getType() === 'entity_reference' &&
        $field_definition->getSetting('target_type') === 'media'
      ) {
        // Check if this media reference field
        // and only allows 'image' media bundles.
        $handler_settings = $field_definition->getSetting('handler_settings');
        if (!empty($handler_settings['target_bundles'])) {
          $media_bundles = $handler_settings['target_bundles'];
          foreach ($media_bundles as $bundle) {
            $media_bundle = $this->entityTypeManager->getStorage('media_type')->load($bundle);
            if ($media_bundle && $media_bundle->get('source') === 'image') {
              $field_options[$field_name] = $field_definition->getLabel();
              break;
            }
          }
        }
      }
    }

    return $field_options;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $csv_file = $form_state->getValue('csv_file');
    if (!empty($csv_file)) {
      $file = $this->entityTypeManager->getStorage('file')->load($csv_file[0]);
      if ($file) {
        $file_path = $this->fileSystem->realpath($file->getFileUri());
        $this->validateCsvFile($file_path, $form_state);
      }
    }
  }

  /**
   * Validate CSV file structure.
   */
  protected function validateCsvFile($file_path, FormStateInterface $form_state) {
    if (($handle = fopen($file_path, 'r')) !== FALSE) {
      $header = fgetcsv($handle);
      fclose($handle);

      $required_columns = ['entity_title', 'image_url'];
      $missing_columns = array_diff($required_columns, $header);

      if (!empty($missing_columns)) {
        $form_state->setErrorByName('csv_file', $this->t('CSV file is missing required columns: @columns', [
          '@columns' => implode(', ', $missing_columns),
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $csv_file_id = $form_state->getValue('csv_file')[0];
    $image_field = $form_state->getValue('image_field');
    $is_multiple = $form_state->getValue('is_multiple');

    $images_options = [
      'entity_type' => 'user',
      'image_field' => $image_field,
      'is_multiple' => $is_multiple,
      'operation' => 'import',
    ];

    // Check if field allows multiple values.
    $field_storage_definition = $this->entityFieldManager
      ->getFieldStorageDefinitions('user')[$image_field];
    $cardinality = $field_storage_definition->getCardinality();

    if (($cardinality === -1 || $cardinality > 1) && $form_state->getValue('is_multiple') != 1) {
      // The field allows multiple values.
      $this->messenger()->addError($this->t("The selected @image_field field allows multiple images. Please check the 'Is Multiple?' box to import multiple images.", ['@image_field' => $image_field]));

      return;
    }

    // Save file as permanent.
    $file = $this->entityTypeManager->getStorage('file')->load($csv_file_id);
    $file->setPermanent();
    $file->save();
    $this->configFactory()->getEditable('user_image_import_form.settings')->set('csv_file', $form_state->getValue('csv_file'))
      ->save();

    $file_path = $this->fileSystem->realpath($file->getFileUri());

    if (($handle = fopen($file_path, 'r')) !== FALSE) {
      $header = fgetcsv($handle);

      $rows = [];
      while (($data = fgetcsv($handle)) !== FALSE) {
        $row = array_combine($header, $data);
        $rows[] = $row;

        // Check if image_field is single value and
        // image_url contains multiple URLs.
        if (
          ($cardinality === 1) &&
          isset($row['image_url']) &&
          strpos($row['image_url'], ',') !== FALSE
        ) {
          // Set error and return.
          $this->messenger()->addError($this->t('Row for entity has multiple image URLs but the selected field only allows a single image.'));
          return;
        }
      }
      fclose($handle);

      // Prepare batch operations.
      $batch_builder = (new BatchBuilder())
        ->setTitle($this->t('Importing images...'))
        ->setInitMessage($this->t('Starting image import...'))
        ->setProgressMessage($this->t('Processed @current out of @total.'))
        ->setErrorMessage($this->t('An error occurred during image import.'))
        ->setFinishCallback([BatchProcessor::class, 'importFinishedCallback']);

      foreach ($rows as $row) {
        $batch_builder->addOperation(
          [BatchProcessor::class, 'processItem'],
          [$row, $images_options]
        );
      }

      batch_set($batch_builder->toArray());
      return;
    }
    else {
      $this->messenger->addError($this->t('Failed to open CSV file.'));
      return;
    }
  }

}
