# Immich Integration

A comprehensive Drupal module that provides service-oriented integration with [Immich](https://immich.app/), the self-hosted photo and video management solution.

## Overview

This module is designed as a **foundation for Immich integration** in Drupal. Rather than providing complete features out-of-the-box, it focuses on offering a robust **service layer** that developers can use to build custom integrations tailored to their specific needs.

The `ImmichClient` service provides comprehensive access to the Immich API, covering:
- Server management and health checks
- Album operations (CRUD)
- Asset management and search
- Timeline and memories
- People (face recognition)
- Tags and shared links
- Libraries and user preferences

## Installation

1. Copy the `immich_integration` folder to `/modules/custom/`
2. Enable the module: `drush en immich_integration`
3. Configure your Immich server connection at `/admin/config/media/immich`

## Configuration

1. Navigate to **Configuration → Media → Immich Integration**
2. Enter your Immich server URL (e.g., `http://localhost:2283`)
3. Enter your Immich API key (generate one in Immich under Account Settings → API Keys)
4. Click "Test Connection" to verify
5. Save configuration

## Service-Oriented Architecture

The core of this module is the `ImmichClient` service (`immich_integration.client`), which provides methods for interacting with the Immich API. Developers can inject this service into their custom modules to build features.

### Basic Usage

```php
// Get the Immich service
$immich = \Drupal::service('immich_integration.client');

// Test connection
$result = $immich->testConnection();
if ($result['success']) {
  // Connection successful
}

// Get all albums
$albums = $immich->getAlbums();
foreach ($albums as $album) {
  echo $album['albumName'];
}

// Get assets from a specific album
$album = $immich->getAlbum($album_id);
foreach ($album['assets'] as $asset) {
  $thumbnail = $immich->getAssetThumbnailUrl($asset['id']);
  echo "<img src='$thumbnail' />";
}
```

## Available Service Methods

### Server Operations

```php
// Test connection
$immich->testConnection();

// Get server version
$immich->getServerVersion();

// Get server configuration
$immich->getServerConfig();

// Get enabled features
$immich->getServerFeatures();

// Get server statistics
$immich->getServerStats();
```

### Album Operations

```php
// List all albums
$immich->getAlbums();

// Get specific album with assets
$immich->getAlbum($album_id);

// Create new album
$immich->createAlbum('Vacation 2024', 'Summer trip', [$asset_id1, $asset_id2]);

// Update album
$immich->updateAlbum($album_id, ['albumName' => 'New Name']);

// Delete album
$immich->deleteAlbum($album_id);

// Add assets to album
$immich->addAssetsToAlbum($album_id, [$asset_id1, $asset_id2]);

// Remove assets from album
$immich->removeAssetsFromAlbum($album_id, [$asset_id1]);
```

### Asset Operations

```php
// Get all assets
$immich->getAssets();

// Get favorite assets only
$immich->getAssets(['isFavorite' => true]);

// Get specific asset
$immich->getAsset($asset_id);

// Update asset
$immich->updateAsset($asset_id, ['isFavorite' => true, 'description' => 'Beach sunset']);

// Delete assets
$immich->deleteAssets([$asset_id1, $asset_id2]);

// Get asset statistics
$immich->getAssetStatistics();

// Get asset URLs (for display)
$thumbnail_url = $immich->getAssetThumbnailUrl($asset_id, 'thumbnail');
$preview_url = $immich->getAssetThumbnailUrl($asset_id, 'preview');
$original_url = $immich->getAssetUrl($asset_id);
```

### Search Operations

```php
// Search assets by metadata
$results = $immich->searchAssets([
  'city' => 'Paris',
  'make' => 'Canon',
  'takenAfter' => '2024-01-01',
]);

// Get places
$places = $immich->searchPlaces();

// Get explore/curated content
$explore = $immich->getExploreData();
```

### Timeline Operations

```php
// Get timeline buckets (monthly/yearly counts)
$buckets = $immich->getTimelineBuckets(['size' => 'MONTH']);

// Get assets for a specific time bucket
$assets = $immich->getTimelineBucket([
  'size' => 'MONTH',
  'timeBucket' => '2024-01-01T00:00:00.000Z',
]);
```

### People Operations (Face Recognition)

```php
// Get all recognized people
$people = $immich->getPeople();

// Get specific person
$person = $immich->getPerson($person_id);

// Update person name
$immich->updatePerson($person_id, ['name' => 'John Doe']);
```

### Tag Operations

```php
// Get all tags
$tags = $immich->getTags();

// Create a tag
$tag = $immich->createTag('Vacation');

// Tag assets
$immich->tagAssets($tag_id, [$asset_id1, $asset_id2]);
```

### Shared Link Operations

```php
// Get all shared links
$links = $immich->getSharedLinks();

// Create shared link for an album
$link = $immich->createSharedLink([
  'type' => 'ALBUM',
  'albumId' => $album_id,
]);

// Delete shared link
$immich->deleteSharedLink($link_id);
```

### User Operations

```php
// Get current user info
$user = $immich->getCurrentUser();

// Get user preferences
$prefs = $immich->getUserPreferences();
```

### Library Operations

```php
// Get all libraries
$libraries = $immich->getLibraries();

// Get library statistics
$stats = $immich->getLibraryStatistics($library_id);
```

### Memory Operations

```php
// Get memories (on this day, etc.)
$memories = $immich->getMemories();
```

## Example: Building a Custom Block

Here's an example of using the service to create a custom block that displays recent photos:

```php
<?php

namespace Drupal\my_module\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\immich_integration\Service\ImmichClient;

/**
 * Provides a 'Recent Photos' block.
 *
 * @Block(
 *   id = "immich_recent_photos",
 *   admin_label = @Translation("Immich Recent Photos"),
 * )
 */
class RecentPhotosBlock extends BlockBase implements ContainerFactoryPluginInterface {

  protected $immichClient;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, ImmichClient $immich_client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->immichClient = $immich_client;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('immich_integration.client')
    );
  }

  public function build() {
    $assets = $this->immichClient->getAssets();
    $recent_assets = array_slice($assets, 0, 10);

    $items = [];
    foreach ($recent_assets as $asset) {
      $items[] = [
        '#theme' => 'image',
        '#uri' => $this->immichClient->getAssetThumbnailUrl($asset['id'], 'thumbnail'),
        '#alt' => $asset['originalFileName'] ?? 'Photo',
      ];
    }

    return [
      '#theme' => 'item_list',
      '#items' => $items,
    ];
  }
}
```

## Example: Custom Gallery Page

```php
<?php

namespace Drupal\my_module\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\immich_integration\Service\ImmichClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

class GalleryController extends ControllerBase {

  protected $immichClient;

  public function __construct(ImmichClient $immich_client) {
    $this->immichClient = $immich_client;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('immich_integration.client')
    );
  }

  public function timeline() {
    // Get timeline buckets
    $buckets = $this->immichClient->getTimelineBuckets(['size' => 'MONTH']);

    $build = [];
    foreach ($buckets as $bucket) {
      $month = date('F Y', strtotime($bucket['timeBucket']));
      $count = $bucket['count'];

      $build[] = [
        '#markup' => "<h3>$month ($count photos)</h3>",
      ];

      // Get assets for this bucket
      $assets = $this->immichClient->getTimelineBucket([
        'size' => 'MONTH',
        'timeBucket' => $bucket['timeBucket'],
      ]);

      foreach ($assets as $asset) {
        $build[] = [
          '#theme' => 'image',
          '#uri' => $this->immichClient->getAssetThumbnailUrl($asset['id']),
          '#alt' => $asset['originalFileName'],
        ];
      }
    }

    return $build;
  }
}
```

## Included Demo Pages

The module includes basic demo pages to help you get started:

- **Settings**: `/admin/config/media/immich` - Configure connection
- **Browse Albums**: `/admin/config/media/immich/albums` - View your albums
- **Album View**: `/admin/config/media/immich/album/{id}` - View photos in an album

These pages demonstrate basic usage of the service and can serve as reference implementations.

## Error Handling

All service methods return `NULL` on failure and log errors to the Drupal watchdog system under the `immich_integration` channel. Always check for NULL returns:

```php
$albums = $immich->getAlbums();
if ($albums === NULL) {
  // Handle error - check logs at /admin/reports/dblog
  \Drupal::messenger()->addError('Failed to fetch albums');
  return;
}
```

## Authentication

The module uses Immich's API key authentication. The API key is passed in the `x-api-key` header with every request. Make sure your API key has appropriate permissions for the operations you need.

## Performance Considerations

- The module makes real-time API calls to your Immich server
- Consider implementing caching in your custom code for frequently accessed data
- Thumbnail URLs require authentication, so they may not work in all contexts
- For production use, consider implementing an image proxy

## Extending the Module

This module is intentionally minimal to serve as a foundation. Common extensions include:

1. **Caching Layer**: Add Drupal cache integration to reduce API calls
2. **Media Integration**: Import Immich assets into Drupal's Media library
3. **Views Integration**: Create Views plugins for Immich data
4. **Field Formatters**: Display Immich images in content types
5. **Image Proxy**: Proxy Immich images through Drupal for better browser compatibility
6. **Batch Operations**: Implement batch API for bulk imports
7. **Webhooks**: Listen for Immich events and sync data

## Requirements

- Drupal 9.4, 10, or 11
- PHP 7.4 or higher
- Access to an Immich server
- Immich API key with appropriate permissions

## API Documentation

For complete Immich API documentation, visit: https://api.immich.app/

## Support

This module provides the service layer only. For feature requests or custom development, extend the module in your own custom code using the provided service methods.

## License

GPL-2.0-or-later

## Credits

- Immich: https://immich.app/
- Immich API Documentation: https://api.immich.app/
