<?php

namespace Drupal\immich_integration\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\immich_integration\Service\ImmichClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying Immich albums.
 */
class ImmichAlbumsController extends ControllerBase {

    /**
     * The Immich client service.
     *
     * @var \Drupal\immich_integration\Service\ImmichClient
     */
    protected $immichClient;

    /**
     * Constructs an ImmichAlbumsController object.
     *
     * @param \Drupal\immich_integration\Service\ImmichClient $immich_client
     *   The Immich client service.
     */
    public function __construct(ImmichClient $immich_client) {
        $this->immichClient = $immich_client;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container) {
        return new static(
            $container->get('immich_integration.client')
        );
    }

    /**
     * Display list of Immich albums.
     *
     * @return array
     *   Render array.
     */
    public function listAlbums() {
        $albums = $this->immichClient->getAlbums();

        if ($albums === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch albums from Immich server. Please check your configuration and server logs.'));
            return [
                '#markup' => $this->t('Unable to connect to Immich server.'),
            ];
        }

        if (empty($albums)) {
            return [
                '#markup' => $this->t('No albums found in your Immich library.'),
            ];
        }

        $rows = [];
        foreach ($albums as $album) {
            $asset_count = isset($album['assetCount']) ? $album['assetCount'] : 0;
            $album_name = $album['albumName'] ?? 'Untitled';
            $album_id = $album['id'] ?? '';

            // Create a link to view the album
            $view_link = [
                '#type' => 'link',
                '#title' => $album_name,
                '#url' => \Drupal\Core\Url::fromRoute('immich_integration.album_view', ['album_id' => $album_id]),
            ];

            $rows[] = [
                'name' => ['data' => $view_link],
                'asset_count' => $asset_count,
                'id' => $album_id,
                'created' => isset($album['createdAt']) ? date('Y-m-d H:i', strtotime($album['createdAt'])) : 'Unknown',
            ];
        }

        $header = [
            $this->t('Album Name'),
            $this->t('Photos/Videos'),
            $this->t('Album ID'),
            $this->t('Created'),
        ];

        return [
            '#type' => 'table',
            '#header' => $header,
            '#rows' => $rows,
            '#empty' => $this->t('No albums found.'),
            '#caption' => $this->t('Albums from your Immich server'),
        ];
    }

    /**
     * View photos from a specific album.
     *
     * @param string $album_id
     *   The album ID.
     *
     * @return array
     *   Render array.
     */
    public function viewAlbum($album_id) {
        $album = $this->immichClient->getAlbum($album_id);

        if ($album === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch album from Immich server.'));
            return [
                '#markup' => $this->t('Unable to load album.'),
            ];
        }

        $album_name = $album['albumName'] ?? 'Untitled Album';
        $assets = $album['assets'] ?? [];

        if (empty($assets)) {
            return [
                '#markup' => $this->t('This album is empty.'),
            ];
        }

        // Limit to first 50 assets for performance
        $assets = array_slice($assets, 0, 50);

        $items = [];
        foreach ($assets as $asset) {
            $asset_id = $asset['id'] ?? '';
            $thumbnail_url = $this->immichClient->getAssetThumbnailUrl($asset_id, 'thumbnail');
            $original_url = $this->immichClient->getAssetUrl($asset_id);
            $filename = $asset['originalFileName'] ?? 'Unknown';

            $items[] = [
                '#theme' => 'image',
                '#uri' => $thumbnail_url,
                '#alt' => $filename,
                '#title' => $filename,
                '#attributes' => [
                    'class' => ['immich-thumbnail'],
                    'style' => 'width: 200px; height: 200px; object-fit: cover; margin: 5px;',
                ],
                '#prefix' => '<a href="' . $original_url . '" target="_blank">',
                '#suffix' => '</a>',
            ];
        }

        return [
            '#type' => 'container',
            '#attributes' => ['class' => ['immich-album-view']],
            'title' => [
                '#markup' => '<h2>' . $this->t('Album: @name', ['@name' => $album_name]) . '</h2>',
            ],
            'info' => [
                '#markup' => '<p>' . $this->t('Showing @count of @total photos/videos', [
                    '@count' => count($items),
                    '@total' => count($album['assets'] ?? []),
                ]) . '</p>',
            ],
            'back_link' => [
                '#type' => 'link',
                '#title' => $this->t('← Back to Albums'),
                '#url' => \Drupal\Core\Url::fromRoute('immich_integration.albums'),
                '#attributes' => ['class' => ['button']],
            ],
            'gallery' => [
                '#type' => 'container',
                '#attributes' => ['style' => 'display: flex; flex-wrap: wrap;'],
                'items' => $items,
            ],
        ];
    }

}
