<?php

namespace Drupal\immich_integration\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\immich_integration\Service\ImmichClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Immich Integration settings.
 */
class ImmichSettingsForm extends ConfigFormBase {

    /**
     * The Immich client service.
     *
     * @var \Drupal\immich_integration\Service\ImmichClient
     */
    protected $immichClient;

    /**
     * Constructs an ImmichSettingsForm object.
     *
     * @param \Drupal\immich_integration\Service\ImmichClient $immich_client
     *   The Immich client service.
     */
    public function __construct(ImmichClient $immich_client) {
        $this->immichClient = $immich_client;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container) {
        return new static(
            $container->get('immich_integration.client')
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames() {
        return ['immich_integration.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId() {
        return 'immich_integration_settings';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state) {
        $config = $this->config('immich_integration.settings');

        // Place connection result at the top
        $form['connection_result'] = [
            '#type' => 'container',
            '#attributes' => ['id' => 'connection-test-result'],
            '#weight' => -100,
        ];

        $form['server_url'] = [
            '#type' => 'url',
            '#title' => $this->t('Immich Server URL'),
            '#description' => $this->t('The base URL of your Immich server (e.g., http://localhost:2283 or https://immich.example.com). Do not include a trailing slash.'),
            '#default_value' => $config->get('server_url'),
            '#required' => TRUE,
        ];

        $form['api_key'] = [
            '#type' => 'textfield',
            '#title' => $this->t('API Key'),
            '#description' => $this->t('Your Immich API key. You can generate one in your Immich account settings under "API Keys".'),
            '#default_value' => $config->get('api_key'),
            '#required' => TRUE,
            '#attributes' => [
                'autocomplete' => 'off',
            ],
        ];

        $form['test_connection'] = [
            '#type' => 'button',
            '#value' => $this->t('Test Connection'),
            '#ajax' => [
                'callback' => '::testConnectionAjax',
                'wrapper' => 'connection-test-result',
                'progress' => [
                    'type' => 'throbber',
                    'message' => $this->t('Testing connection...'),
                ],
            ],
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * AJAX callback for testing the connection.
     */
    public function testConnectionAjax(array &$form, FormStateInterface $form_state) {
        // Temporarily save the form values to test with them
        $temp_config = $this->config('immich_integration.settings');
        $server_url = $form_state->getValue('server_url');
        $api_key = $form_state->getValue('api_key');

        // Save temporarily
        $temp_config->set('server_url', $server_url);
        $temp_config->set('api_key', $api_key);
        $temp_config->save();

        $result = $this->immichClient->testConnection();

        if ($result['success']) {
            $form['connection_result']['#markup'] = '<div class="messages messages--status">' . $result['message'] . '</div>';

            // Try to get server version
            $server_info = $this->immichClient->getServerVersion();
            if ($server_info) {
                $form['connection_result']['#markup'] .= '<div class="messages messages--status">Server Version: ' . ($server_info['version'] ?? 'Unknown') . '</div>';
            }
        } else {
            $form['connection_result']['#markup'] = '<div class="messages messages--error">' . $result['message'] . '</div>';
        }

        return $form['connection_result'];
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state) {
        $this->config('immich_integration.settings')
            ->set('server_url', rtrim($form_state->getValue('server_url'), '/'))
            ->set('api_key', $form_state->getValue('api_key'))
            ->save();

        parent::submitForm($form, $form_state);
    }

}
