<?php

namespace Drupal\immich_integration\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for communicating with Immich API.
 *
 * This service provides a comprehensive interface to the Immich API,
 * allowing developers to build custom integrations and features.
 * All methods return NULL on failure and log errors for debugging.
 */
class ImmichClient
{

    /**
     * The HTTP client.
     *
     * @var \GuzzleHttp\ClientInterface
     */
    protected $httpClient;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs an ImmichClient object.
     *
     * @param \GuzzleHttp\ClientInterface $http_client
     *   The HTTP client.
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
     *   The logger factory.
     */
    public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory)
    {
        $this->httpClient = $http_client;
        $this->configFactory = $config_factory;
        $this->logger = $logger_factory->get('immich_integration');
    }

    /**
     * Get the Immich server URL from configuration.
     *
     * @return string
     *   The server URL.
     */
    protected function getServerUrl()
    {
        $config = $this->configFactory->get('immich_integration.settings');
        return rtrim($config->get('server_url'), '/');
    }

    /**
     * Get the API key from configuration.
     *
     * @return string
     *   The API key.
     */
    protected function getApiKey()
    {
        $config = $this->configFactory->get('immich_integration.settings');
        return $config->get('api_key');
    }

    /**
     * Make a request to the Immich API.
     *
     * @param string $method
     *   The HTTP method (GET, POST, PUT, DELETE, etc.).
     * @param string $endpoint
     *   The API endpoint (e.g., '/api/albums').
     * @param array $options
     *   Additional request options (body, query params, etc.).
     *
     * @return array|null
     *   The decoded JSON response or NULL on failure.
     */
    protected function request($method, $endpoint, array $options = [])
    {
        $server_url = $this->getServerUrl();
        $api_key = $this->getApiKey();

        if (empty($server_url) || empty($api_key)) {
            $this->logger->error('Immich server URL or API key not configured.');
            return NULL;
        }

        $url = $server_url . $endpoint;

        // Add API key to headers
        $options['headers']['x-api-key'] = $api_key;
        $options['headers']['Accept'] = 'application/json';

        try {
            $response = $this->httpClient->request($method, $url, $options);
            $body = $response->getBody()->getContents();
            return json_decode($body, TRUE);
        } catch (GuzzleException $e) {
            $this->logger->error('Immich API request failed: @message', [
                '@message' => $e->getMessage(),
            ]);
            return NULL;
        }
    }

    // =========================================================================
    // SERVER ENDPOINTS
    // =========================================================================

    /**
     * Test the connection to the Immich server.
     *
     * @return array
     *   Array with 'success' boolean and 'message' string.
     */
    public function testConnection()
    {
        $result = $this->request('GET', '/api/server/ping');

        if ($result !== NULL) {
            return [
                'success' => TRUE,
                'message' => 'Successfully connected to Immich server.',
                'data' => $result,
            ];
        }

        return [
            'success' => FALSE,
            'message' => 'Failed to connect to Immich server. Please check your settings.',
        ];
    }

    /**
     * Get server version information.
     *
     * @return array|null
     *   Server version info or NULL on failure.
     */
    public function getServerVersion()
    {
        return $this->request('GET', '/api/server/version');
    }

    /**
     * Get server configuration and features.
     *
     * @return array|null
     *   Server config or NULL on failure.
     */
    public function getServerConfig()
    {
        return $this->request('GET', '/api/server/config');
    }

    /**
     * Get server features (e.g., face recognition, search).
     *
     * @return array|null
     *   Server features or NULL on failure.
     */
    public function getServerFeatures()
    {
        return $this->request('GET', '/api/server/features');
    }

    /**
     * Get server statistics.
     *
     * @return array|null
     *   Server statistics or NULL on failure.
     */
    public function getServerStats()
    {
        return $this->request('GET', '/api/server/statistics');
    }

    // =========================================================================
    // ALBUM ENDPOINTS
    // =========================================================================

    /**
     * Get all albums.
     *
     * @param array $params
     *   Optional query parameters (e.g., ['shared' => true]).
     *
     * @return array|null
     *   Array of albums or NULL on failure.
     */
    public function getAlbums(array $params = [])
    {
        $options = [];
        if (!empty($params)) {
            $options['query'] = $params;
        }
        return $this->request('GET', '/api/albums', $options);
    }

    /**
     * Get a specific album by ID.
     *
     * @param string $album_id
     *   The album ID.
     *
     * @return array|null
     *   Album data or NULL on failure.
     */
    public function getAlbum($album_id)
    {
        return $this->request('GET', '/api/albums/' . $album_id);
    }

    /**
     * Create a new album.
     *
     * @param string $album_name
     *   The album name.
     * @param string|null $description
     *   Optional album description.
     * @param array $asset_ids
     *   Optional array of asset IDs to add to the album.
     *
     * @return array|null
     *   Created album data or NULL on failure.
     */
    public function createAlbum($album_name, $description = NULL, array $asset_ids = [])
    {
        $data = ['albumName' => $album_name];
        if ($description) {
            $data['description'] = $description;
        }
        if (!empty($asset_ids)) {
            $data['assetIds'] = $asset_ids;
        }

        return $this->request('POST', '/api/albums', [
            'json' => $data,
        ]);
    }

    /**
     * Update an album.
     *
     * @param string $album_id
     *   The album ID.
     * @param array $data
     *   Album data to update (albumName, description, etc.).
     *
     * @return array|null
     *   Updated album data or NULL on failure.
     */
    public function updateAlbum($album_id, array $data)
    {
        return $this->request('PATCH', '/api/albums/' . $album_id, [
            'json' => $data,
        ]);
    }

    /**
     * Delete an album.
     *
     * @param string $album_id
     *   The album ID.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function deleteAlbum($album_id)
    {
        return $this->request('DELETE', '/api/albums/' . $album_id);
    }

    /**
     * Add assets to an album.
     *
     * @param string $album_id
     *   The album ID.
     * @param array $asset_ids
     *   Array of asset IDs to add.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function addAssetsToAlbum($album_id, array $asset_ids)
    {
        return $this->request('PUT', '/api/albums/' . $album_id . '/assets', [
            'json' => ['ids' => $asset_ids],
        ]);
    }

    /**
     * Remove assets from an album.
     *
     * @param string $album_id
     *   The album ID.
     * @param array $asset_ids
     *   Array of asset IDs to remove.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function removeAssetsFromAlbum($album_id, array $asset_ids)
    {
        return $this->request('DELETE', '/api/albums/' . $album_id . '/assets', [
            'json' => ['ids' => $asset_ids],
        ]);
    }

    // =========================================================================
    // ASSET ENDPOINTS
    // =========================================================================

    /**
     * Get all assets with optional filtering.
     *
     * @param array $params
     *   Query parameters for filtering (e.g., ['isFavorite' => true]).
     *
     * @return array|null
     *   Array of assets or NULL on failure.
     */
    public function getAssets(array $params = [])
    {
        $options = [];
        if (!empty($params)) {
            $options['query'] = $params;
        }
        return $this->request('GET', '/api/assets', $options);
    }

    /**
     * Get a specific asset by ID.
     *
     * @param string $asset_id
     *   The asset ID.
     *
     * @return array|null
     *   Asset data or NULL on failure.
     */
    public function getAsset($asset_id)
    {
        return $this->request('GET', '/api/assets/' . $asset_id);
    }

    /**
     * Update an asset.
     *
     * @param string $asset_id
     *   The asset ID.
     * @param array $data
     *   Asset data to update (isFavorite, description, etc.).
     *
     * @return array|null
     *   Updated asset data or NULL on failure.
     */
    public function updateAsset($asset_id, array $data)
    {
        return $this->request('PUT', '/api/assets/' . $asset_id, [
            'json' => $data,
        ]);
    }

    /**
     * Delete assets.
     *
     * @param array $asset_ids
     *   Array of asset IDs to delete.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function deleteAssets(array $asset_ids)
    {
        return $this->request('DELETE', '/api/assets', [
            'json' => ['ids' => $asset_ids],
        ]);
    }

    /**
     * Get asset statistics.
     *
     * @return array|null
     *   Asset statistics or NULL on failure.
     */
    public function getAssetStatistics()
    {
        return $this->request('GET', '/api/assets/statistics');
    }

    /**
     * Get asset thumbnail URL.
     *
     * Note: This returns a URL, not data. The URL requires authentication.
     *
     * @param string $asset_id
     *   The asset ID.
     * @param string $size
     *   Thumbnail size (preview or thumbnail).
     *
     * @return string
     *   The thumbnail URL.
     */
    public function getAssetThumbnailUrl($asset_id, $size = 'preview')
    {
        $server_url = $this->getServerUrl();
        return $server_url . '/api/assets/' . $asset_id . '/thumbnail?size=' . $size;
    }

    /**
     * Get asset original file URL.
     *
     * Note: This returns a URL, not data. The URL requires authentication.
     *
     * @param string $asset_id
     *   The asset ID.
     *
     * @return string
     *   The download URL.
     */
    public function getAssetUrl($asset_id)
    {
        $server_url = $this->getServerUrl();
        return $server_url . '/api/assets/' . $asset_id . '/original';
    }

    // =========================================================================
    // SEARCH ENDPOINTS
    // =========================================================================

    /**
     * Search assets by metadata.
     *
     * @param array $search_params
     *   Search parameters (city, country, make, model, etc.).
     *
     * @return array|null
     *   Search results or NULL on failure.
     */
    public function searchAssets(array $search_params)
    {
        return $this->request('POST', '/api/search/metadata', [
            'json' => $search_params,
        ]);
    }

    /**
     * Search for places.
     *
     * @return array|null
     *   Array of places or NULL on failure.
     */
    public function searchPlaces()
    {
        return $this->request('GET', '/api/search/places');
    }

    /**
     * Get explore data (curated content).
     *
     * @return array|null
     *   Explore data or NULL on failure.
     */
    public function getExploreData()
    {
        return $this->request('GET', '/api/search/explore');
    }

    // =========================================================================
    // TIMELINE ENDPOINTS
    // =========================================================================

    /**
     * Get timeline buckets (asset counts per time period).
     *
     * @param array $params
     *   Query parameters (size, timeBucket, etc.).
     *
     * @return array|null
     *   Timeline buckets or NULL on failure.
     */
    public function getTimelineBuckets(array $params = [])
    {
        $options = [];
        if (!empty($params)) {
            $options['query'] = $params;
        }
        return $this->request('GET', '/api/timeline/buckets', $options);
    }

    /**
     * Get assets for a specific timeline bucket.
     *
     * @param array $params
     *   Query parameters (size, timeBucket, etc.).
     *
     * @return array|null
     *   Assets in the bucket or NULL on failure.
     */
    public function getTimelineBucket(array $params)
    {
        return $this->request('GET', '/api/timeline/bucket', [
            'query' => $params,
        ]);
    }

    // =========================================================================
    // PEOPLE ENDPOINTS (Face Recognition)
    // =========================================================================

    /**
     * Get all people (faces).
     *
     * @param bool $with_hidden
     *   Include hidden people.
     *
     * @return array|null
     *   Array of people or NULL on failure.
     */
    public function getPeople($with_hidden = FALSE)
    {
        return $this->request('GET', '/api/people', [
            'query' => ['withHidden' => $with_hidden],
        ]);
    }

    /**
     * Get a specific person by ID.
     *
     * @param string $person_id
     *   The person ID.
     *
     * @return array|null
     *   Person data or NULL on failure.
     */
    public function getPerson($person_id)
    {
        return $this->request('GET', '/api/people/' . $person_id);
    }

    /**
     * Update a person.
     *
     * @param string $person_id
     *   The person ID.
     * @param array $data
     *   Person data to update (name, etc.).
     *
     * @return array|null
     *   Updated person data or NULL on failure.
     */
    public function updatePerson($person_id, array $data)
    {
        return $this->request('PUT', '/api/people/' . $person_id, [
            'json' => $data,
        ]);
    }

    // =========================================================================
    // SHARED LINK ENDPOINTS
    // =========================================================================

    /**
     * Get all shared links.
     *
     * @return array|null
     *   Array of shared links or NULL on failure.
     */
    public function getSharedLinks()
    {
        return $this->request('GET', '/api/shared-links');
    }

    /**
     * Create a shared link.
     *
     * @param array $data
     *   Shared link data (type, albumId, assetIds, etc.).
     *
     * @return array|null
     *   Created shared link or NULL on failure.
     */
    public function createSharedLink(array $data)
    {
        return $this->request('POST', '/api/shared-links', [
            'json' => $data,
        ]);
    }

    /**
     * Delete a shared link.
     *
     * @param string $link_id
     *   The shared link ID.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function deleteSharedLink($link_id)
    {
        return $this->request('DELETE', '/api/shared-links/' . $link_id);
    }

    // =========================================================================
    // TAG ENDPOINTS
    // =========================================================================

    /**
     * Get all tags.
     *
     * @return array|null
     *   Array of tags or NULL on failure.
     */
    public function getTags()
    {
        return $this->request('GET', '/api/tags');
    }

    /**
     * Create a tag.
     *
     * @param string $tag_name
     *   The tag name.
     *
     * @return array|null
     *   Created tag or NULL on failure.
     */
    public function createTag($tag_name)
    {
        return $this->request('POST', '/api/tags', [
            'json' => ['name' => $tag_name],
        ]);
    }

    /**
     * Tag assets.
     *
     * @param string $tag_id
     *   The tag ID.
     * @param array $asset_ids
     *   Array of asset IDs to tag.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function tagAssets($tag_id, array $asset_ids)
    {
        return $this->request('PUT', '/api/tags/' . $tag_id . '/assets', [
            'json' => ['ids' => $asset_ids],
        ]);
    }

    // =========================================================================
    // USER ENDPOINTS
    // =========================================================================

    /**
     * Get current user information.
     *
     * @return array|null
     *   User data or NULL on failure.
     */
    public function getCurrentUser()
    {
        return $this->request('GET', '/api/users/me');
    }

    /**
     * Get user preferences.
     *
     * @return array|null
     *   User preferences or NULL on failure.
     */
    public function getUserPreferences()
    {
        return $this->request('GET', '/api/users/me/preferences');
    }

    // =========================================================================
    // LIBRARY ENDPOINTS
    // =========================================================================

    /**
     * Get all libraries.
     *
     * @return array|null
     *   Array of libraries or NULL on failure.
     */
    public function getLibraries()
    {
        return $this->request('GET', '/api/libraries');
    }

    /**
     * Get library statistics.
     *
     * @param string $library_id
     *   The library ID.
     *
     * @return array|null
     *   Library statistics or NULL on failure.
     */
    public function getLibraryStatistics($library_id)
    {
        return $this->request('GET', '/api/libraries/' . $library_id . '/statistics');
    }

    // =========================================================================
    // MEMORY ENDPOINTS
    // =========================================================================

    /**
     * Get all memories.
     *
     * @return array|null
     *   Array of memories or NULL on failure.
     */
    public function getMemories()
    {
        return $this->request('GET', '/api/memories');
    }

}
