<?php

namespace Drupal\inactive_users\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\user\Entity\User;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 *
 */
class InactiveUsersController extends ControllerBase {

  /**
   *
   */
  public function list() {
    $cutoff = strtotime('-6 months');
    $delete_cutoff = strtotime('-7 days');
    $now = \Drupal::time()->getCurrentTime();

    $uids = \Drupal::entityQuery('user')
      ->condition('status', 1)
      ->condition('uid', 0, '>')
      ->notExists('login')
      ->accessCheck(FALSE)
      ->execute();

    $uids_with_login = \Drupal::entityQuery('user')
      ->condition('status', 1)
      ->condition('uid', 0, '>')
      ->condition('login', $cutoff, '<')
      ->accessCheck(FALSE)
      ->execute();

    $uids = array_merge($uids, $uids_with_login);
    $users = User::loadMultiple($uids);

    $rows = [];
    foreach ($users as $user) {
      $login_time = $user->getLastLoginTime();
      $last_login_display = $login_time ? \Drupal::service('date.formatter')->format($login_time, 'short') : 'Never';

      // Check if warning sent and 7 days have passed.
      $warning_field = $user->get('field_inactive_warning_sent');
      $warning_sent = $warning_field->value ? strtotime($warning_field->value) : 0;

      if ($warning_sent && $login_time < $warning_sent && $now - $warning_sent >= 7 * 24 * 60 * 60) {
        // User didn’t log in after 7 days delete.
        $user->delete();
        continue;
      }

      // Create "Send Email" link.
      if ($warning_sent) {
        $send_link = ['#markup' => 'Already Sent'];
      }
      else {
        $url = Url::fromRoute('inactive_users.send_email', ['user' => $user->id()]);
        $send_link = Link::fromTextAndUrl('Send Email', $url)->toRenderable();
      }

      $rows[] = [
        $user->id(),
        $user->getAccountName(),
        $user->getEmail(),
        $last_login_display,
        $warning_sent ? \Drupal::service('date.formatter')->format($warning_sent, 'short') : ['data' => $send_link],
      ];

    }

    $header = ['UID', 'Username', 'Email', 'Last Login', 'Action'];

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => 'No inactive users found.',
    ];
  }

  /**
   *
   */
  public function sendEmail($user) {
    $account = User::load($user);

    if (!$account || !$account->isActive()) {
      $this->messenger()->addError("Invalid user.");
      return new RedirectResponse('/admin/people/inactive-users');
    }

    $warning_sent = $account->get('field_inactive_warning_sent')->value;

    if ($warning_sent) {
      $this->messenger()->addWarning("Warning email already sent to {$account->getAccountName()}.");
      return new RedirectResponse('/admin/people/inactive-users');
    }

    $email = $account->getEmail();
    $username = $account->getAccountName();

    $mailManager = \Drupal::service('plugin.manager.mail');
    $params = ['username' => $username];
    $langcode = \Drupal::languageManager()->getDefaultLanguage()->getId();

    $result = $mailManager->mail('inactive_users', 'warning', $email, $langcode, $params);

    if ($result['result']) {
      $account->set('field_inactive_warning_sent', date('Y-m-d\TH:i:s'));
      $account->save();
      $this->messenger()->addStatus("Email sent to $username.");
    }
    else {
      $this->messenger()->addError("Failed to send email.");
    }

    return new RedirectResponse('/admin/people/inactive-users');
  }

}
