<?php

declare(strict_types=1);

namespace Drupal\inline_image_saver\EventSubscriber;

use Drupal\Component\Plugin\Discovery\CachedDiscoveryInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\inline_image_saver\Form\InlineImageSaverSettingsForm;

/**
 * A subscriber invalidating caches when validation setting is changed.
 */
class InlineImageSaverSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected readonly FieldTypePluginManagerInterface $fieldTypePluginManager,
    protected readonly TypedDataManagerInterface $typedDataManager,
  ) {}

  /**
   * Invalidates caches when validation setting is changed.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The configuration event.
   *
   * @see \Drupal\inline_image_saver\Hook\InlineImageSaverHooks::fieldInfoAlter()
   */
  public function onConfigSave(ConfigCrudEvent $event): void {
    if ($event->getConfig()->getName() === InlineImageSaverSettingsForm::CONFIG_NAME && $event->isChanged('enable_validation')) {
      if ($this->fieldTypePluginManager instanceof CachedDiscoveryInterface) {
        $this->fieldTypePluginManager->clearCachedDefinitions();
      }
      $this->typedDataManager->clearCachedDefinitions();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[ConfigEvents::SAVE][] = ['onConfigSave'];
    return $events;
  }

}
