<?php

declare(strict_types=1);

namespace Drupal\inline_image_saver;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\inline_image_saver\MimeType\InlineImageMimeResolverInterface;

/**
 * Provides an image MIME type resolver that delegates to a chain of resolvers.
 */
class InlineImageMime implements InlineImageMimeInterface {
  use StringTranslationTrait;

  /**
   * The resolver with the highest priority.
   */
  protected ?InlineImageMimeResolverInterface $preferredResolver = NULL;

  /**
   * The registered resolvers grouped by priority.
   *
   * @var array<int, \Drupal\inline_image_saver\MimeType\InlineImageMimeResolverInterface[]>
   */
  protected array $resolvers = [];

  /**
   * The sorted array of supported resolvers, ready for resolution.
   *
   * @var \Drupal\inline_image_saver\MimeType\InlineImageMimeResolverInterface[]|null
   */
  protected ?array $preparedResolvers = NULL;

  /**
   * {@inheritdoc}
   */
  public function isSupported(?TranslatableMarkup &$reason = NULL): bool {
    $reason = NULL;
    $this->prepareMimeResolvers();
    if ($this->preparedResolvers) {
      return TRUE;
    }
    $this->preferredResolver?->isSupported($reason);
    $reason ??= $this->t('No MIME type resolvers available.');
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function resolveByUri(string $uri, ?string &$mime_type = NULL): bool {
    $this->prepareMimeResolvers();
    foreach ($this->preparedResolvers as $resolver) {
      if ($resolver->resolveByUri($uri, $mime_type)) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function resolveByData(string $data, ?string &$mime_type = NULL): bool {
    $this->prepareMimeResolvers();
    foreach ($this->preparedResolvers as $resolver) {
      if ($resolver->resolveByData($data, $mime_type)) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function addMimeResolver(InlineImageMimeResolverInterface $resolver, int $priority = 0): static {
    $this->resolvers[$priority][] = $resolver;
    $this->preparedResolvers = NULL;
    return $this;
  }

  /**
   * Sorts resolvers by priority and filters out unsupported ones.
   */
  protected function prepareMimeResolvers(): void {
    if ($this->preparedResolvers !== NULL) {
      return;
    }
    $this->preparedResolvers = [];
    $this->preferredResolver = NULL;
    if (!$this->resolvers) {
      return;
    }
    krsort($this->resolvers);
    $this->preferredResolver = reset($this->resolvers)[0];
    foreach ($this->resolvers as $resolvers) {
      foreach ($resolvers as $resolver) {
        if ($resolver->isSupported()) {
          $this->preparedResolvers[] = $resolver;
        }
      }
    }
  }

}
