<?php

declare(strict_types=1);

namespace Drupal\inline_image_saver;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\file\FileInterface;
use Drupal\inline_image_saver\Struct\InlineImageData;
use Drupal\inline_image_saver\Struct\InlineImageValidation;
use Drupal\text\Plugin\Field\FieldType\TextItemBase;

/**
 * Defines an interface for managing inline images in text content.
 */
interface InlineImageSaverInterface {

  /**
   * The default upload directory name.
   *
   * @var string
   */
  public const DEFAULT_UPLOAD_DIRECTORY_NAME = 'inline-image-saver';

  /**
   * Symbol used to identify placeholder keys in template strings.
   *
   * @var string
   */
  public const PLACEHOLDER_SYMBOL = '@';

  /**
   * The hash algorithm used for generating unique image identifiers.
   *
   * @var string
   */
  public const HASH_ALGO = 'xxh64';

  /**
   * Checks if a field should be processed for inline images.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface|class-string $field_definition_or_class
   *   The field definition or field type class-string to check.
   *
   * @return bool
   *   TRUE if the field needs processing, FALSE otherwise.
   */
  public function isFieldTypeProcessable(FieldDefinitionInterface|string $field_definition_or_class): bool;

  /**
   * Checks if an entity should be processed for inline images.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if the entity needs processing, FALSE otherwise.
   */
  public function isEntityProcessable(FieldableEntityInterface $entity): bool;

  /**
   * Processes inline images within an entity's text fields.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity to process.
   *
   * @return bool
   *   TRUE if any changes were made to the entity, FALSE otherwise.
   */
  public function processEntity(FieldableEntityInterface $entity): bool;

  /**
   * Checks if a text field item should be processed for inline images.
   *
   * @param \Drupal\text\Plugin\Field\FieldType\TextItemBase $item
   *   The text field item to check.
   *
   * @return bool
   *   TRUE if the item needs processing, FALSE otherwise.
   */
  public function isTextItemProcessable(TextItemBase $item): bool;

  /**
   * Processes inline images within a single text field item.
   *
   * @param \Drupal\text\Plugin\Field\FieldType\TextItemBase $item
   *   The text field item to process.
   *
   * @return bool
   *   TRUE if any changes were made to the field item, FALSE otherwise.
   */
  public function processTextItem(TextItemBase $item): bool;

  /**
   * Parses a text field item's value into a DOM document if it contains images.
   *
   * @param \Drupal\text\Plugin\Field\FieldType\TextItemBase $item
   *   The text field item to parse.
   *
   * @return \DOMDocument|null
   *   The parsed DOM document, or NULL if no images found or parsing failed.
   */
  public function parseTextItemDom(TextItemBase $item): ?\DOMDocument;

  /**
   * Updates a text field item's value with the serialized DOM document.
   *
   * @param \Drupal\text\Plugin\Field\FieldType\TextItemBase $item
   *   The text field item to update.
   * @param \DOMDocument $value
   *   The DOM document to serialize.
   */
  public function serializeTextItemDom(TextItemBase $item, \DOMDocument $value): void;

  /**
   * Validates an inline image element against configured rules.
   *
   * @param \DOMElement $img
   *   The <img> element to validate.
   *
   * @return \Drupal\inline_image_saver\Struct\InlineImageValidation
   *   The validation result.
   *
   * @throws \InvalidArgumentException
   *   If the element is not an <img> tag.
   */
  public function validateImage(\DOMElement $img): InlineImageValidation;

  /**
   * Downloads an image from its src attribute (URL or data URI).
   *
   * @param \DOMElement $img
   *   The <img> element containing the image source.
   *
   * @return \Drupal\inline_image_saver\Struct\InlineImageData|null
   *   Object containing the image data and metadata,
   *   or NULL if download failed.
   *
   * @throws \InvalidArgumentException
   *   If the element is not an <img> tag.
   */
  public function downloadImage(\DOMElement $img): ?InlineImageData;

  /**
   * Creates a temporary file entity from image data.
   *
   * @param \Drupal\inline_image_saver\Struct\InlineImageData $data
   *   The image data and metadata.
   * @param string $directory
   *   The directory where to save the file.
   *
   * @return \Drupal\file\FileInterface
   *   The created temporary file entity.
   *
   * @throws \Drupal\Core\File\Exception\FileException
   *   Implementation may throw FileException or its subtype on failure.
   */
  public function saveImage(InlineImageData $data, string $directory): FileInterface;

  /**
   * Generates a unique filename for image.
   *
   * @param \Drupal\inline_image_saver\Struct\InlineImageData $data
   *   The image data and metadata.
   *
   * @return string
   *   A safe and unique filename.
   */
  public function generateImageFilename(InlineImageData $data): string;

  /**
   * Returns the upload directory URI for the given text field item.
   *
   * Ensures that the directory exists before returning the URI.
   *
   * @param \Drupal\text\Plugin\Field\FieldType\TextItemBase $item
   *   The text field item.
   *
   * @return string
   *   The URI of the upload directory.
   */
  public function getTextItemUploadDirectory(TextItemBase $item): string;

  /**
   * Returns the default upload directory URI.
   *
   * Ensures that the directory exists before returning the URI.
   *
   * @return string
   *   The URI of the default upload directory.
   */
  public function getDefaultUploadDirectory(): string;

  /**
   * Checks if a source string is a data URI image.
   *
   * @param string $src
   *   The source string to check.
   *
   * @return bool
   *   TRUE if the source is a data URI, FALSE otherwise.
   */
  public function isDataUriImage(string $src): bool;

  /**
   * Extracts binary data from a data URI string.
   *
   * @param string $src
   *   The data URI string.
   *
   * @return string|null
   *   The decoded binary data, or NULL if decoding failed.
   */
  public function extractDataUriImage(string $src): ?string;

  /**
   * Checks if text contains placeholder tokens.
   *
   * @param string $text
   *   The text to check.
   *
   * @return bool
   *   TRUE if placeholders are found, FALSE otherwise.
   */
  public function containsPlaceholders(string $text): bool;

  /**
   * Creates placeholder tokens from a DOM element's attributes.
   *
   * @param \DOMElement $element
   *   The element to extract attributes from.
   *
   * @return array<string, string>
   *   Array mapping placeholder tokens to escaped attribute values.
   */
  public function placeholdersFromElement(\DOMElement $element): array;

  /**
   * Replaces placeholder tokens in text with values.
   *
   * @param string $markup
   *   The text containing placeholder tokens.
   * @param array<string, string> $placeholders
   *   Array mapping placeholder tokens to replacement values.
   *
   * @return string
   *   The text with placeholders replaced.
   */
  public function processPlaceholders(string $markup, array $placeholders): string;

  /**
   * Replaces a DOM node with new HTML markup.
   *
   * @param \DOMNode $node
   *   The DOM node to replace.
   * @param string $markup
   *   The HTML markup to insert instead of the node.
   *
   * @return \DOMDocumentFragment|null
   *   The created fragment, or NULL if markup parsing failed.
   */
  public function replaceNodeWithMarkup(\DOMNode $node, string $markup): ?\DOMDocumentFragment;

  /**
   * Returns settings or a subset of settings.
   *
   * @return array<string, bool|string|array|null>
   *   The subset of settings if $key is provided, or all settings as an array.
   */
  public function getSettings(): array;

  /**
   * Returns a specific setting by name.
   *
   * @param string|null $name
   *   (optional) The setting name to retrieve.
   *
   * @return bool|string|array|null
   *   The setting value, or NULL if not found.
   */
  public function getSetting(?string $name = NULL): bool|string|array|null;

  /**
   * Clears the cached settings.
   */
  public function clearCachedSettings(): void;

  /**
   * Returns the upload directory URI for the given text field item.
   *
   * Ensures that the directory exists before returning the URI.
   *
   * @param \Drupal\text\Plugin\Field\FieldType\TextItemBase $item
   *   The text field item.
   *
   * @return string
   *   The URI of the upload directory.
   *
   * @phpcs:ignore Drupal.Commenting.Deprecated.DeprecatedMissingSeeTag
   * @deprecated in inline_image_saver:2.1.0 and is removed from drupal:3.0.0.
   * Use ::getTextItemUploadDirectory() instead.
   */
  public function prepareUploadDirectory(TextItemBase $item): string;

}
