<?php

declare(strict_types=1);

namespace Drupal\inline_image_saver\MimeType;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\Mime\Exception\LogicException;

/**
 * A base image MIME type resolver.
 */
abstract class InlineImageMimeResolverBase implements InlineImageMimeResolverInterface {
  use StringTranslationTrait;

  /**
   * Resolves a MIME type from a file URI.
   *
   * @param string $uri
   *   The file URI to analyze.
   *
   * @return string|null
   *   The resolved MIME type or NULL on failure.
   */
  abstract protected function doResolveByUri(string $uri): ?string;

  /**
   * Resolves a MIME type from binary data.
   *
   * @param string $data
   *   The binary data to analyze.
   *
   * @return string|null
   *   The resolved MIME type or NULL on failure.
   */
  abstract protected function doResolveByData(string $data): ?string;

  /**
   * {@inheritdoc}
   */
  public function resolveByUri(string $uri, ?string &$mime_type = NULL): bool {
    return $this->doResolveMimeType(fn () => $this->doResolveByUri($uri), $mime_type);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveByData(string $data, ?string &$mime_type = NULL): bool {
    return $this->doResolveMimeType(fn () => $this->doResolveByData($data), $mime_type);
  }

  /**
   * Resolves and validates the MIME type using a callable.
   *
   * @param callable(): (string|null) $resolver
   *   A callable that returns a MIME type string or NULL.
   * @param string|null $mime_type
   *   Populated with the resolved MIME type.
   *
   * @return bool
   *   TRUE if the resolved MIME type indicates an image, FALSE otherwise.
   *
   * @throws \Symfony\Component\Mime\Exception\LogicException
   *   If the resolver is not supported.
   */
  protected function doResolveMimeType(callable $resolver, ?string &$mime_type): bool {
    $mime_type = NULL;
    if (!$this->isSupported()) {
      throw new LogicException(\sprintf('The "%s" resolver is not supported.', static::class));
    }
    try {
      $mime_type = $resolver();
    }
    catch (\Throwable) {
      return FALSE;
    }
    return $mime_type && str_starts_with($mime_type, 'image/');
  }

}
