<?php

declare(strict_types=1);

namespace Drupal\inline_image_saver\Struct;

use Drupal\file\FileInterface;

/**
 * Value object representing an inline image validation result.
 */
class InlineImageValidation {

  /**
   * Constructs the object.
   *
   * @param \DOMElement $img
   *   The source <img> element.
   * @param \Drupal\inline_image_saver\Struct\InlineImageError|null $error
   *   The validation error, or NULL if validation passed.
   * @param \Drupal\file\FileInterface|null $file
   *   The associated image file entity, or NULL if not available.
   * @param bool $isDataUriImage
   *   TRUE if the image uses a data URI scheme, FALSE otherwise.
   * @param array<string, mixed> $context
   *   (optional) Additional validation context.
   * @param string|null $message
   *   (optional) A custom error message, or NULL to use the default.
   */
  public function __construct(
    public readonly \DOMElement $img,
    public readonly ?InlineImageError $error,
    public readonly ?FileInterface $file,
    public readonly bool $isDataUriImage,
    public array $context = [],
    public ?string $message = NULL,
  ) {}

  /**
   * Creates a successful validation result.
   *
   * @param \DOMElement $img
   *   The source <img> element.
   * @param \Drupal\file\FileInterface|null $file
   *   (optional) The associated image file entity, or NULL if not available.
   * @param bool $is_data_uri
   *   (optional) TRUE if the image uses a data URI scheme, FALSE otherwise.
   * @param array<string, mixed> $context
   *   (optional) Additional validation context.
   *
   * @return static
   *   The validation result object.
   */
  public static function ok(\DOMElement $img, ?FileInterface $file = NULL, bool $is_data_uri = FALSE, array $context = []): static {
    return new static(
      img: $img,
      error: NULL,
      file: $file,
      isDataUriImage: $is_data_uri,
      context: $context,
      message: NULL,
    );
  }

  /**
   * Creates a failed validation result.
   *
   * @param \DOMElement $img
   *   The source <img> element.
   * @param \Drupal\inline_image_saver\Struct\InlineImageError $error
   *   The validation error.
   * @param \Drupal\file\FileInterface|null $file
   *   (optional) The associated image file entity, or NULL if not available.
   * @param bool $is_data_uri
   *   (optional) TRUE if the image uses a data URI scheme, FALSE otherwise.
   * @param array<string, mixed> $context
   *   (optional) Additional validation context.
   * @param string|null $message
   *   (optional) A custom error message, or NULL to use the default.
   *
   * @return static
   *   The validation result object.
   */
  public static function error(
    \DOMElement $img,
    InlineImageError $error,
    ?FileInterface $file = NULL,
    bool $is_data_uri = FALSE,
    array $context = [],
    ?string $message = NULL,
  ): static {
    return new static(
      img: $img,
      error: $error,
      file: $file,
      isDataUriImage: $is_data_uri,
      context: $context,
      message: $message,
    );
  }

}
