Inline SVG
==============

The **Inline SVG** module provides a custom Drupal field type that allows
you tostore and render raw SVG code directly within your content. This
field type supports editing via a textarea widget with built-in validation
to prevent security risks, and it renders inline SVG safely on your site.

This module is ideal for sites that need flexible, reusable SVG graphics
without relying on external files or image styles.

* * *

Table of Contents
-----------------

*   [Requirements](#requirements)
*   [Installation](#installation)
*   [Usage](#usage)
*   [Why Use Inline SVG Field Instead of SVG File Upload?](#why-use-inline-svg)
*   [Inline SVG Media Submodule](#svg-code-field-media-submodule)
*   [Configuration](#configuration)
*   [Hooks](#hooks)
*   [Maintainers](#maintainers)

* * *

Requirements
------------

This module requires Drupal Core 10 or 11.

* * *

Installation
------------

Install the module as usual for contributed Drupal modules.

If you use Composer, run:

    composer require drupal/inline_svg

Enable the module with Drush:

    drush en inline_svg

* * *

Usage
-----

After enabling the module, you can add the `SVG Code` field type to
any content type or entity that supports fields. Editors can input
raw SVG code directly into a textarea, which will be validated for
unsafe code and then rendered inline.

* * *

Why Use Inline SVG Field Instead of SVG File Upload?
----------------------------------------------------

*   **Inline Editing and Dynamic Control**  
    Edit SVG code directly in the textarea or widget without
    re-uploading files for every change. Copy-paste SVG code
    from design tools like Figma or Illustrator.
    _Advantage:_ Real-time control without file reprocessing.
*   **Style Overrides Per Display**  
    Use the SVG code formatter from Manage Display to override
    styles like fill, stroke, font-family, height, width,
    transform etc for different view modes (e.g., teasers,
    blocks, full view) — without changing the original
    SVG code. _Advantage:_ Centralized visual control.
*   **No External HTTP Requests**  
    Inline SVGs embed directly into HTML, avoiding extra
    HTTP requests needed for file-based SVGs.
    _Advantage:_ Better site performance and reduced latency.
*   **Security with Controlled Rendering**  
    Inline SVG field sanitizes SVG content and disallows
    risky tags like `<script>`, external links, or event
        handlers. Avoids file system risks associated with
        SVG uploads. _Advantage:_ Reduced attack surface.
*   **Reusable with Per-Instance Styling**  
    Reuse the same SVG code in multiple entities with different
    style overrides, unlike static file uploads which require
    duplication. _Advantage:_ One codebase, many appearances.
*   **Smooth Integration with Twig/Templates**  
    Since SVG renders as HTML, you can use Twig conditions
    and CSS styling easily. _Advantage:_ Cleaner theming
    and full frontend control.
*   **No File Storage or Cleanup Needed**  
    SVGs are stored as field content, so no files clutter
    the filesystem or media library.
    _Advantage:_ Lightweight content management.
*   **Designer-Developer Workflow**  
    Designers can hand off raw SVG code without extra
    file prep, and developers can control rendering securely
    with formatter settings.
    _Advantage:_ Fasterdesign-to-production turnaround.
*   **Better for Theming and Accessibility**  
    Easily inject accessibility attributes like
    `<title>`, `<desc>`, `aria-label`, or `role="img"`
        inline._Advantage:_ Improved accessibility compliance.
*   **Better Site Building Experience**  
    Use inline SVG fields in Paragraphs, Layout Builder,
    blocks, or custom content types with per-display configuration.
    Editors don’t need SVG or CSS knowledge.
    _Advantage:_ Editor-friendly and layout-ready.


**Ideal use cases:** Icons, logos, illustrations, flags,
themed visuals with dynamic color overrides, accessible
inline visuals, and reusable illustrations with varied
styling per section, views or page.

* * *

Inline SVG Media Submodule
------------------------------

The **Inline SVG Media** submodule extends the main
`inline_svg` module by integrating the Inline SVG into
Drupal's Media system.

**Important:** To use SVG code as reusable media items,
you must install both modules:

*   `inline_svg` (main module)
*   `inline_svg_media` (submodule)

The media submodule provides:

*   A media source plugin for raw SVG code.
*   A custom media library add form for SVG code input.
*   Configurable display settings for media entities
with SVG content.

To install the media submodule, enable it after the main module:

    drush en inline_svg_media

Make sure that the `inline_svg_media` module depends
on `inline_svg` and the core `field` module, so the
field type and dependencies are available.

* * *
Hooks
-----

The `inline_svg` module provides the following hooks
to allow other modules to customize SVG validation and filtering behavior.

### hook\_svg\_dangerous\_patterns\_alter(array &$patterns)

Alter the list of dangerous patterns used for SVG code validation.

Other modules can implement this hook to add, remove,
or modify the patterns that are considered unsafe in SVG code.
This is useful for tightening or relaxing validation rules
based on your site's security policies.

**Parameters:**

*   `&patterns`: An associative array where keys are dangerous
pattern strings (e.g., `'<script'`) and values are human-readable
explanations of why they are disallowed.

**Example usage:**

    function mymodule_svg_dangerous_patterns_alter(array &$patterns) {
      // Add a new dangerous pattern.
      $patterns['data:'] = 'Data URLs are not allowed in SVG code.';
    
      // Modify an existing pattern message.
      if (isset($patterns['<script'])) {
        $patterns['<script'] = 'Custom message for <script> tags.';
      }
    
      // Remove a dangerous pattern.
      unset($patterns['onclick']);
    }
    

### hook\_svg\_allowed\_tags\_alter(array &$allowed\_tags)

Alter the list of allowed SVG tags used during XSS filtering of SVG code.

Other modules can implement this hook to add new allowed tags,
remove existing ones, or otherwise customize which SVG tags are permitted.

**Parameters:**

*   `&allowed_tags`: An array of allowed SVG tag names (strings).

**Example usage:**

    function mymodule_svg_allowed_tags_alter(array &$allowed_tags) {
      // Add a new allowed SVG tag.
      $allowed_tags[] = 'filter';
    
      // Remove an allowed tag.
      if (($key = array_search('switch', $allowed_tags)) !== FALSE) {
        unset($allowed_tags[$key]);
      }
    
      // Ensure the list contains unique tags only.
      $allowed_tags = array_unique($allowed_tags);
    }

* * *

Configuration
-------------

Once installed, you can configure media types to use the SVG Code
media source. When adding media, editors can input SVG markup
directly, which will be validated and safely rendered inline
wherever the media entity is used.

For details on configuring fields and media displays, refer
to Drupal's standard field and media management interfaces.

After installing the module, you can configure how the
**SVG Code** field widget behaves by visiting the
**Manage form display** tab for any content type or entity using the field.

Under the widget settings (gear ⚙️ icon next to the field),
you can enable or disable:

*   **Preview SVG Link** – Allows editors to preview the
rendered SVG directly from the form.
*   **Download SVG Link** – Provides a download button to
export the SVG code as a `.svg` file.
*  **Filename for download** – Allows customizing the default
filename (e.g., `icon.svg`) when the download button is clicked.

These options allow site builders to customize the editorial
interface based on specific use cases or roles.

#### To configure:

1.  Go to _Structure → Content types → \[Your Content Type\]
→ Manage form display_.
2.  Click the **⚙️ gear icon** next to the **SVG Code** field.
3.  Adjust the **Preview SVG Link** , **Download SVG Link**
and **Filename for download** options as needed..
4.  Save the form display configuration.

**Note:** These settings are per-field-instance, allowing
different behaviors in different content types or forms.

### Formatter Configuration (Manage Display)

The **SVG Renderer with Style Overrides** formatter allows site
builders to apply styling and attribute overrides to the `<svg>`tag
when rendering SVG code. This is especially useful for theming,
accessibility, and adapting visuals for different view modes like
Teasers, Full content, or Blocks.

To configure the formatter:

1. Go to **Structure → Content types → [Your content type] → Manage display**
2. Find your Inline SVG in the list.
3. Choose **SVG Renderer with Style Overrides** as the formatter.
4. Click the **gear ⚙️ icon** next to the formatter to open
configuration options.

The following override options are available:

- `fill`: Fill color (e.g., `#000`, `red`)
- `fill-opacity`: Opacity of the fill (0–1)
- `stroke`: Stroke color
- `stroke-width`: Stroke width (e.g., `2px`)
- `stroke-opacity`: Opacity of the stroke
- `stroke-linecap`: Linecap style (`butt`, `round`, `square`)
- `stroke-linejoin`: Line join style (`miter`, `round`, `bevel`)
- `stroke-dasharray`: Dash pattern (e.g., `5,5`)
- `stroke-dashoffset`: Dash offset value
- `font-family`: Font family for text inside SVG
- `font-size`: Font size (e.g., `16px`)
- `font-weight`: Font weight (e.g., `bold`, `400`)
- `text-anchor`: Text alignment (`start`, `middle`, `end`)
- `letter-spacing`: Spacing between letters
- `width`: Override the width of the SVG (e.g., `100px`, `100%`)
- `height`: Override the height of the SVG (e.g., `auto`)
- `transform`: Apply transformations (e.g., `rotate(45)`, `scale(1.5)`)
- `class`: Additional CSS classes to append to the `<svg>` tag

These settings are applied dynamically during rendering by
modifying the attributes of the `<svg>` tag. You can set
different values for different view modes to visually
differentiate how the same SVG appears in different contexts.

This approach gives site builders powerful theming control
without altering the original SVG code.

* * *

Maintainers
-----------

**Sujan Shrestha** – [sujan shrestha](https://www.drupal.org/u/sujan-shrestha)
