<?php

namespace Drupal\inline_svg\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'svg_code_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "svg_code_formatter",
 *   label = @Translation("SVG Renderer with Style Overrides"),
 *   field_types = {
 *     "svg_code"
 *   }
 * )
 */
class SvgCodeFormatter extends FormatterBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a SvgCodeFormatter object.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The settings array.
   * @param string $label
   *   The label.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Third party settings.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct($plugin_id, $plugin_definition, $field_definition, array $settings, $label, $view_mode, array $third_party_settings, ModuleHandlerInterface $module_handler) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'fill' => '',
      'fill_opacity' => '',
      'stroke' => '',
      'stroke_width' => '',
      'stroke_opacity' => '',
      'stroke_linecap' => '',
      'stroke_linejoin' => '',
      'stroke_dasharray' => '',
      'stroke_dashoffset' => '',
      'font_family' => '',
      'font_size' => '',
      'font_weight' => '',
      'text_anchor' => '',
      'letter_spacing' => '',
      'height' => '',
      'width' => '',
      'transform' => '',
      'class' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $fields = [
      'fill' => 'Fill color (e.g., #000000 or red)',
      'fill_opacity' => 'Fill opacity (0–1)',
      'stroke' => 'Stroke color (e.g., #FF0000)',
      'stroke_width' => 'Stroke width (e.g., 2px)',
      'stroke_opacity' => 'Stroke opacity (0–1)',
      'stroke_linecap' => 'Linecap (butt, round, square)',
      'stroke_linejoin' => 'Linejoin (miter, round, bevel)',
      'stroke_dasharray' => 'Dash pattern (e.g., 5,5)',
      'stroke_dashoffset' => 'Dash offset',
      'font_family' => 'Font family (e.g., Arial, sans-serif)',
      'font_size' => 'Font size (e.g., 16px)',
      'font_weight' => 'Font weight (e.g., bold, 400)',
      'text_anchor' => 'Text anchor (start, middle, end)',
      'letter_spacing' => 'Letter spacing (e.g., 1px)',
      'width' => 'Override width (e.g., 100px, 100%)',
      'height' => 'Override height (e.g., 100px, auto)',
      'transform' => 'Transform (e.g., rotate(45), scale(1.5))',
      'class' => 'Additional CSS classes to append',
    ];

    $elements = [];
    foreach ($fields as $key => $description) {
      $formatted_key = ucwords(str_replace('_', ' ', $key));

      $elements[$key] = [
        '#type' => 'textfield',
        '#title' => $this->t('@formatted_key', ['@formatted_key' => $formatted_key]),
        '#default_value' => $this->getSetting($key),
        '#description' => $this->t('@description', ['@description' => $description]),
      ];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    return [
      $this->t('Renders inline SVG with optional attribute and style overrides.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $settings = $this->getSettings();
    $allowed_tags = [
        // Container elements.
      'svg', 'g', 'defs', 'symbol', 'use',

        // Descriptive elements.
      'title', 'desc',

        // Shape elements.
      'path', 'rect', 'circle', 'ellipse', 'line', 'polyline', 'polygon',

        // Text elements.
      'text', 'tspan', 'textPath',

        // Gradient and pattern elements.
      'linearGradient', 'radialGradient', 'stop', 'pattern',

        // Clipping and masking.
      'clipPath', 'mask',

        // Image and reference.
      'image',

        // Others.
      'switch', 'a', 'view', 'marker', 'metadata',
    ];

    // Alter allowed tags using the injected module handler.
    $this->moduleHandler->alter('svg_allowed_tags', $allowed_tags);

    foreach ($items as $delta => $item) {
      $svg = $item->svg;
      $svg = Xss::filter($svg, $allowed_tags);

      // Validate.
      if (stripos($svg, '<svg') === FALSE) {
        $elements[$delta] = ['#markup' => $this->t('Invalid SVG code.')];
        continue;
      }

      // Generate style attribute string.
      $style_attributes = '';
      $style_map = [
        'fill', 'fill_opacity', 'stroke', 'stroke_width', 'stroke_opacity',
        'stroke_linecap', 'stroke_linejoin', 'stroke_dasharray', 'stroke_dashoffset',
        'font_family', 'font_size', 'font_weight', 'text_anchor', 'letter_spacing',
      ];
      foreach ($style_map as $key) {
        if (!empty($settings[$key])) {
          $css_key = str_replace('_', '-', $key);
          $style_attributes .= $css_key . ':' . $settings[$key] . ';';
        }
      }

      // Build list of attributes to override.
      $attribute_overrides = [];

      if (!empty($settings['width'])) {
        $attribute_overrides['width'] = $settings['width'];
      }

      if (!empty($settings['height'])) {
        $attribute_overrides['height'] = $settings['height'];
      }

      if (!empty($settings['transform'])) {
        $attribute_overrides['transform'] = $settings['transform'];
      }

      if (!empty($style_attributes)) {
        $attribute_overrides['style'] = $style_attributes;
      }

      // Modify <svg ...> tag with overrides.
      $svg = preg_replace_callback('/<svg\s([^>]*)>/i', function ($matches) use ($attribute_overrides, $settings) {
        $existing_attrs = $matches[1];

        // Parse existing attributes into an associative array.
        preg_match_all('/(\w+)=["\']([^"\']*)["\']/', $existing_attrs, $attr_matches, PREG_SET_ORDER);
        $attr_array = [];
        foreach ($attr_matches as $attr) {
          $attr_array[$attr[1]] = $attr[2];
        }

        // Handle class: append if exists, otherwise add.
        if (!empty($settings['class'])) {
          if (isset($attr_array['class'])) {
            $attr_array['class'] .= ' ' . $settings['class'];
          }
          else {
            $attr_array['class'] = $settings['class'];
          }
        }

        // Apply or override attributes except class.
        foreach ($attribute_overrides as $key => $value) {
          $attr_array[$key] = $value;
        }

        // Rebuild the <svg ...> tag.
        $new_attrs = '';
        foreach ($attr_array as $key => $value) {
          $new_attrs .= ' ' . $key . '="' . htmlspecialchars($value, ENT_QUOTES) . '"';
        }

        return '<svg' . $new_attrs . '>';
        // Only once.
      }, $svg, 1);

      // Return processed SVG.
      $elements[$delta] = [
        '#type' => 'inline_template',
        '#template' => '{{ svg|raw }}',
        '#context' => ['svg' => $svg],
      ];
    }

    return $elements;
  }

}
