<?php

namespace Drupal\inline_svg\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'svg_code' field type.
 *
 * This field type stores raw SVG code as text, allowing it to be used in
 * various places throughout the site.
 *
 * @FieldType(
 *   id = "svg_code",
 *   label = @Translation("SVG Code"),
 *   description = @Translation("Stores raw SVG code."),
 *   default_widget = "svg_code_textarea",
 *   default_formatter = "svg_code_formatter"
 * )
 */
class SvgCodeItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   *
   * Defines the properties for the 'svg_code' field type. The field stores
   * the raw SVG code as a string.
   *
   * @param \Drupal\Core\Field\FieldStorageDefinitionInterface $field_definition
   *   The field storage definition.
   *
   * @return \Drupal\Core\TypedData\DataDefinition[]
   *   An array of data definitions for the field's properties.
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['svg'] = DataDefinition::create('string')
      ->setLabel(t('SVG Code'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   *
   * Defines the database schema for the 'svg_code' field type. The 'svg'
   * column stores the raw SVG code as a text field with a large size.
   *
   * @param \Drupal\Core\Field\FieldStorageDefinitionInterface $field_definition
   *   The field storage definition.
   *
   * @return array
   *   The database schema definition for the field type.
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'svg' => [
          'type' => 'text',
          'size' => 'big',
          'not null' => FALSE,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   *
   * Checks if the 'svg_code' field is empty. The field is considered empty
   * if the SVG value is NULL or an empty string.
   *
   * @return bool
   *   TRUE if the field is empty, FALSE otherwise.
   */
  public function isEmpty() {
    $value = $this->get('svg')->getValue();
    return $value === NULL || $value === '';
  }

}
