<?php

namespace Drupal\inline_svg\Plugin\Field\FieldWidget;

use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;
use Drupal\Component\Utility\Random;

/**
 * Defines the widget for the SVG Code textarea.
 *
 * This widget allows content editors to input raw SVG code via a textarea.
 * It is used to edit the 'svg_code' field type.
 *
 * @FieldWidget(
 *   id = "svg_code_textarea",
 *   label = @Translation("SVG Code textarea"),
 *   field_types = {
 *     "svg_code"
 *   }
 * )
 */
class SvgCodeTextareaWidget extends WidgetBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a SvgCodeTextareaWidget object.
   *
   * @param string $plugin_id
   *   The plugin ID for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Third party settings.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, ModuleHandlerInterface $module_handler) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_preview_link' => TRUE,
      'show_download_link' => TRUE,
      'file_name' => 'svg-icon',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['show_preview_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Preview Link'),
      '#default_value' => $this->getSetting('show_preview_link'),
    ];

    $elements['show_download_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Download Link'),
      '#default_value' => $this->getSetting('show_download_link'),
    ];

    $elements['file_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Filename for download'),
      '#default_value' => $this->getSetting('file_name'),
      '#required' => TRUE,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $summary[] = $this->t('Show Preview Link: @value', ['@value' => $this->getSetting('show_preview_link') ? 'Yes' : 'No']);
    $summary[] = $this->t('Show Download Link: @value', ['@value' => $this->getSetting('show_download_link') ? 'Yes' : 'No']);
    $summary[] = $this->t('Filename for download: @value', ['@value' => $this->getSetting('file_name')]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $random = new Random();
    $rand = $random->machineName();

    $title = $element['#title'] ?? 'SVG Code';

    $class = 'textarea-' . $rand;
    $element['svg'] = [
      '#type' => 'textarea',
      '#title' => $this->t('@title', ['@title' => $title]),
      '#default_value' => $items[$delta]->svg ?? '',
      '#description' => $this->t('Enter the full SVG markup to display on the page.'),
      '#rows' => 10,
      '#attributes' => [
        'class' => [$class],
      ],
    ];

    // Add Preview link if enabled.
    if ($this->getSetting('show_preview_link')) {
      // Add Preview link.
      $element['preview_link'] = [
        '#type' => 'link',
        '#title' => $this->t('Preview SVG'),
        '#url' => Url::fromRoute('<none>'),
        '#attributes' => [
          'class' => ['svg-code-preview-link'],
          'href' => '#',
          'data-textarea-class' => "{$class}",
        ],
        '#attached' => [
          'library' => [
            'inline_svg/svg_preview_modal',
          ],
        ],
      ];
    }

    // Add Download link if enabled.
    if ($this->getSetting('file_name')) {
      // Add Download link.
      $element['download_link'] = [
        '#type' => 'link',
        '#title' => $this->t('Download SVG'),
        '#url' => Url::fromRoute('<none>'),
        '#attributes' => [
          'class' => ['svg-code-download-link'],
          'href' => '#',
          'data-textarea-class' => "{$class}",
          'download' => $this->getSetting('file_name'),
          'style' => 'margin-left: 10px;',
        ],
        '#attached' => [
          'library' => [
            'inline_svg/svg_preview_modal',
          ],
        ],
      ];
    }

    // Add the element validation callback.
    $element['#element_validate'][] = [$this, 'validateElement'];

    return $element;
  }

  /**
   * Validates the SVG code input.
   */
  public function validateElement(&$element, FormStateInterface $form_state, &$complete_form) {
    $svg_code = $element['svg']['#value'] ?? '';

    if (!empty($svg_code)) {
      $dangerous_patterns = [
        '<script' => 'Script tags like <script> are not allowed for security reasons.',
        '<?php' => 'PHP code is not allowed inside SVG files.',
        'onload' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onLoad' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onclick' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onClick' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onerror' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onError' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onmouseover' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onMouseOver' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onmouseout' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'onMouseOut' => 'Event handler attributes (e.g., onload, onclick) are not allowed in SVG code.',
        'javascript:' => 'JavaScript URLs (e.g., javascript:) are not allowed in SVG code.',
      ];

      // Allow other modules to alter the patterns.
      $this->moduleHandler->alter('svg_dangerous_patterns', $dangerous_patterns);

      foreach ($dangerous_patterns as $pattern => $message) {
        if (strpos($svg_code, $pattern) !== FALSE) {
          $form_state->setError($element, $this->t('Invalid SVG content detected: @message', ['@message' => $message]));
          return;
        }
      }
    }
  }

}
