<?php

namespace Drupal\inqube\EventSubscriber;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\inqube\Ajax\ElasticsearchDebugCommand;
use Drupal\inqube\Plugin\views\query\Elasticsearch;
use Drupal\views\Ajax\ViewAjaxResponse;
use Drupal\views\ViewExecutable;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Subscribes and adds Elasticsearch query debug command to Ajax responses.
 */
class AjaxResponseSubscriber implements EventSubscriberInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * AjaxResponseSubscriber constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(AccountInterface $current_user, ConfigFactoryInterface $config_factory) {
    $this->currentUser = $current_user;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [KernelEvents::RESPONSE => 'onResponse'];
  }

  /**
   * Prints Elasticsearch query to console for debugging.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event) {
    // Do nothing if use does not have permission to administer views.
    if (!$this->currentUser->hasPermission('administer views')) {
      return;
    }

    // Do nothing if "Show SQL query" setting is disabled.
    if (!$this->configFactory->get('views.settings')->get('ui.show.sql_query.enabled')) {
      return;
    }

    // Get response.
    $response = $event->getResponse();

    if ($response instanceof ViewAjaxResponse) {
      // Get view.
      $view = $response->getView();

      if ($view->ajaxEnabled() && $queryHandler = $view->getQuery()) {
        if ($queryHandler instanceof Elasticsearch) {
          $this->addDebugCommand($response, $view);
        }
      }
    }
  }

  /**
   * Adds debug command to response.
   */
  protected function addDebugCommand(AjaxResponse $response, ViewExecutable $view) {
    $response->addAttachments(['library' => ['inqube/debug']]);
    $response->addCommand(new ElasticsearchDebugCommand($view->build_info['query']));
  }

}
