<?php

namespace Drupal\inqube\Plugin\views\query;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\elasticsearch_helper\ElasticsearchClientVersion;
use Drupal\inqube\ElasticsearchQueryBuilderManager;
use Drupal\views\Plugin\views\query\QueryPluginBase;
use Drupal\views\ResultRow;
use Drupal\views\ViewExecutable;
use Elastic\Elasticsearch\Client;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views query plugin for an Elasticsearch query.
 *
 * @ingroup views_query_plugins
 *
 * @ViewsQuery(
 *   id = "elasticsearch_query",
 *   title = @Translation("Elasticsearch Query"),
 *   help = @Translation("Query will be generated and run using the Elasticsearch API.")
 * )
 */
class Elasticsearch extends QueryPluginBase {

  /**
   * A list of tables in the order they should be added, keyed by alias.
   */
  protected $tableQueue = [];

  /**
   * Holds an array of tables and counts added so that we can create aliases.
   */
  public $tables = [];

  /**
   * Holds an array of relationships.
   *
   * These are aliases of the primary table that represent different ways to
   * join the same table in.
   */
  public $relationships = [];

  /**
   * An array of sections of the WHERE query.
   *
   * Each section is in itself an array of pieces and a flag as to whether or
   * not it should be AND or OR.
   */

  public $where = [];
  /**
   * An array of sections of the HAVING query.
   *
   * Each section is in itself an array of pieces and a flag as to whether or
   * not it should be AND or OR.
   */
  public $having = [];

  /**
   * A simple array of order by clauses.
   */
  public $orderby = [];

  /**
   * A simple array of group by clauses.
   */
  public $groupby = [];


  /**
   * An array of fields.
   */
  public $fields = [];

  /**
   * A flag as to whether or not to make the primary field distinct.
   *
   * @var bool
   */
  public $distinct = FALSE;

  /**
   * @var bool
   */
  protected $hasAggregate = FALSE;

  /**
   * Should this query be optimized for counts, for example no sorts.
   */
  protected $getCountOptimized = NULL;

  /**
   * An array mapping table aliases and field names to field aliases.
   */
  protected $fieldAliases = [];

  /**
   * Query tags which will be passed over to the dbtng query object.
   */
  public $tags = [];

  /**
   * Is the view marked as not distinct.
   *
   * @var bool
   */
  protected $noDistinct;

  /**
   * The database-specific date handler.
   *
   * @var \Drupal\views\Plugin\views\query\DateSqlInterface
   */
  protected $dateSql;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The count field definition.
   */
  // phpcs:ignore Drupal.NamingConventions.ValidVariableName.LowerCamelName
  public array $count_field;

  /**
   * Client.
   *
   * @var \Elastic\Elasticsearch\Client
   */
  protected $elasticsearchClient;

  /**
   * Entity manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Query builder manager.
   *
   * @var \Drupal\inqube\ElasticsearchQueryBuilderManager
   */
  protected $elasticsearchQueryBuilderManager;

  /**
   * Query builder.
   *
   * @var \Drupal\inqube\ElasticsearchQueryBuilderInterface
   */
  protected $queryBuilder;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, Client $elasticsearch_client, EntityTypeManagerInterface $entity_type_manager, ElasticsearchQueryBuilderManager $query_builder_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->elasticsearchClient = $elasticsearch_client;
    $this->entityTypeManager = $entity_type_manager;
    $this->elasticsearchQueryBuilderManager = $query_builder_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('elasticsearch_helper.elasticsearch_client'),
      $container->get('entity_type.manager'),
      $container->get('elasticsearch_query_builder.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function setOptionDefaults(array &$storage, array $options) {
    parent::setOptionDefaults($storage, $options);
    $storage['query_builder'] = '';
    $storage['entity_relationship'] = [
      'entity_type_key' => '',
      'entity_id_key' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $query_builder_options = [];
    foreach ($this->elasticsearchQueryBuilderManager->getDefinitions() as $query_builder_plugin) {
      $query_builder_options[$query_builder_plugin['id']] = sprintf('%s (%s)', $query_builder_plugin['label'], $query_builder_plugin['id']);
    }

    $form['query_builder'] = [
      '#type' => 'select',
      '#title' => $this->t('Elasticsearch query builder'),
      '#empty_value' => '',
      '#options' => $query_builder_options,
      '#default_value' => $this->options['query_builder'],
      '#required' => FALSE,
    ];

    $form['entity_relationship'] = [
      '#type' => 'details',
      '#title' => $this->t('Entity relationship'),
      '#description' => $this->t('Define default entity relationship.'),
      '#open' => TRUE,
    ];

    $form['entity_relationship']['entity_type_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity type field'),
      '#description' => $this->t('A field in Elasticsearch results which contains entity type name. To set a fixed value, prefix the string with @ (e.g., @node).'),
      '#default_value' => $this->options['entity_relationship']['entity_type_key'],
    ];

    $form['entity_relationship']['entity_id_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity ID field'),
      '#description' => $this->t('A field in Elasticsearch results which contains entity ID value.'),
      '#default_value' => $this->options['entity_relationship']['entity_id_key'],
      '#group' => 'entity_type_key',
    ];
  }

  /**
   * Builds the necessary info to execute the query.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   View.
   */
  public function build(ViewExecutable $view) {
    // Store the view in the object to be able to use it later.
    $this->view = $view;

    // Initiate pager.
    $view->initPager();

    // Let the pager modify the query to add limits.
    $view->pager->query();

    $view->build_info['query'] = $this->query();
  }

  /**
   * Returns an empty array as there's no physical table in Elasticsearch.
   *
   * @param string $table
   *   Table.
   * @param mixed $relationship
   *   Relationship.
   *
   * @return string
   *   Empty string.
   */
  public function ensureTable($table, $relationship = NULL) {
    return '';
  }

  /**
   * Returns the field as is as there's no need to limit fields in result set.
   *
   * @param string $table
   *   Table.
   * @param string $field
   *   Field.
   * @param string $alias
   *   Alias.
   * @param array $params
   *   Params.
   *
   * @return mixed
   *   Field.
   */
  public function addField($table, $field, $alias = '', array $params = []) {
    return $field;
  }

  /**
   * Placeholder method.
   *
   * @param string $group
   *   Group.
   * @param string $field
   *   Field.
   * @param mixed $value
   *   Value.
   * @param mixed $operator
   *   Operator.
   */
  public function addWhere($group, $field, $value = NULL, $operator = NULL) {
  }

  /**
   * Placeholder method.
   *
   * @param string $group
   *   Group.
   * @param string $snippet
   *   Snippet.
   * @param array $args
   *   Arguments.
   */
  public function addWhereExpression($group, $snippet, array $args = []) {
  }

  /**
   * Placeholder method.
   *
   * @param string $table
   *   Table.
   * @param mixed $field
   *   Field.
   * @param string $order
   *   Order.
   * @param string $alias
   *   Alias.
   * @param array $params
   *   Params.
   */
  public function addOrderBy($table, $field = NULL, $order = 'ASC', $alias = '', array $params = []) {
    $this->orderby[] = [
      'table' => $table,
      'field' => $field,
      'order' => $order,
      'alias' => $alias,
      'params' => $params,
    ];
  }

  /**
   * Placeholder method.
   *
   * @param string $clause
   *   Clause.
   */
  public function addGroupBy($clause) {
  }

  /**
   * Placeholder method.
   */
  public function addRelationship() {
  }

  /**
   * Placeholder method.
   */
  public function placeholder() {
  }

  /**
   * Returns instance of a query builder plugin.
   *
   * @return \Drupal\inqube\ElasticsearchQueryBuilderInterface|null
   *   Query builder.
   */
  public function getQueryBuilder() {
    if ($this->options['query_builder'] && !isset($this->queryBuilder)) {
      try {
        $this->queryBuilder = $this->elasticsearchQueryBuilderManager->createInstance($this->options['query_builder']);
        $options = [];
        // Initialize the plugin.
        $this->queryBuilder->init($this->view, $this->displayHandler, $options);
      }
      catch (\Exception $e) {
        \Drupal::logger('inqube')->error($e->getMessage());
      }
    }

    return $this->queryBuilder;
  }

  /**
   * {@inheritdoc}
   */
  public function query($get_count = FALSE) {
    /** @var \Drupal\inqube\ElasticsearchQueryBuilderInterface $query_builder */
    $query_builder = $this->getQueryBuilder();
    $query = $query_builder->buildQuery();

    // Apply limit and offset to the query.
    $limits = [
      'size' => $this->getLimit(),
      'from' => $this->offset ?? 0,
    ];

    return array_merge($limits, $query);
  }

  /**
   * {@inheritdoc}
   */
  public function validate() {
    $errors = [];

    // Validate query builder settings (on created views only).
    if (!$this->view->storage->isNew() && empty($this->options['query_builder'])) {
      $errors[] = $this->t('Query builder plugin needs to be defined for this view to work. Configure query builder in the query settings.');
    }

    return $errors;
  }

  /**
   * Executes Elasticsearch query and returns the result.
   *
   * @param array $query
   *   Query.
   *
   * @return array
   *   Exection result.
   */
  public function executeQuery(array $query) {
    return $this->elasticsearchClient->search($query);
  }

  /**
   * Returns result row from a search hit.
   *
   * @param array $hit
   *   Hits.
   * @param int $index
   *   Index.
   *
   * @return \Drupal\views\ResultRow
   *   Result row.
   */
  protected function createResultRowFromHit(array $hit, $index) {
    return new ResultRow($hit);
  }

  /**
   * Indexes the result set.
   *
   * @param \Drupal\views\ResultRow[] $result
   *   Index results.
   */
  protected function indexResult(array &$result) {
    foreach ($result as $index => $row) {
      $row->index = $index;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute(ViewExecutable $view) {
    $query = $view->build_info['query'];
    $data = [];
    $result = [];

    try {
      if ($data = $this->executeQuery($query)) {
        $index = 0;
        foreach ($data['hits']['hits'] as $hit) {
          $result[] = $this->createResultRowFromHit($hit, $index);
          $index++;
        }
        // To be able to access the aggregations later on.
        $view->data = $data;
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('inqube')->error($e->getMessage());
    }

    $this->indexResult($result);
    $view->result = $result;

    $view->pager->postExecute($view->result);
    $view->pager->total_items = $this->getTotalHits($data);
    // Account for offset when calculating total results.
    if (!empty($view->pager->options['offset'])) {
      // Make sure that total is never negative.
      if ($view->pager->total_items >= $view->pager->options['offset']) {
        $view->pager->total_items -= $view->pager->options['offset'];
      }
      else {
        $view->pager->total_items = 0;
      }
    }
    $view->pager->updatePageInfo();
    $view->total_rows = $view->pager->getTotalItems();

    // Load all entities contained in the results.
    $this->loadEntities($result);
  }

  /**
   * Returns nested value from the Elasticsearch result.
   *
   * Examples:
   *
   * - "id" will return value of element "id".
   * - "_source.id" will return a value of _source][id].
   * - "@node" will return "node" (value will not be determined).
   *
   * @param string $key
   *   Key.
   * @param array|object $data
   *   Data.
   * @param string $separator
   *   Separator.
   * @param string $default
   *   Default.
   *
   * @return mixed|null
   *   Value.
   */
  protected function getNestedValue($key, $data, $separator = '.', $default = NULL) {
    // If $key starts with a @, it means that the key should be returned as a
    // string and there's no need to look for a value.
    if (isset($key[0]) && $key[0] == '@') {
      return substr($key, 1);
    }

    if (!is_array($data) && !is_object($data)) {
      return NULL;
    }

    // Cast $data into an array so that $data can be processed with NestedArray.
    if (is_object($data)) {
      $data = (array) $data;
    }

    $parts = explode($separator, $key);

    if (count($parts) == 1) {
      return $data[$key] ?? $default;
    }
    else {
      $value = NestedArray::getValue($data, $parts, $key_exists);
      return $key_exists ? $value : $default;
    }
  }

  /**
   * Returns a list of entity relationship information, keyed.
   *
   * Also only valid relationship information is returned (i.e., with defined
   * entity type and entity ID keys).
   *
   * @return array
   *   Relationships.
   *
   * @see hook_views_data_alter()
   */
  public function getEntityRelationships() {
    $result = [];

    foreach ($this->displayHandler->getHandlers('relationship') as $handler_id => $handler) {
      if (isset($handler->options['entity_type_key'], $handler->options['entity_id_key'])) {
        $result[$handler_id] = [
          'entity_type_key' => $handler->options['entity_type_key'],
          'entity_id_key' => $handler->options['entity_id_key'],
        ];
      }
    }

    return ['none' => $this->options['entity_relationship']] + $result;
  }

  /**
   * {@inheritdoc}
   */
  public function loadEntities(&$results) {
    $entity_relationships = $this->getEntityRelationships();

    // No entity tables found, nothing else to do here.
    if (empty($entity_relationships)) {
      return;
    }

    $entity_types = array_keys($this->entityTypeManager->getDefinitions());
    $entity_ids_by_type = [];

    foreach ($entity_relationships as $relationship_id => $info) {
      foreach ($results as $index => $result) {
        // Get entity type value from result.
        $entity_type = $this->getNestedValue($info['entity_type_key'], $result);

        if (isset($entity_type) && in_array($entity_type, $entity_types)) {
          // Get entity ID value from result.
          $entity_id = $this->getNestedValue($info['entity_id_key'], $result);

          if (isset($entity_id)) {
            $entity_ids_by_type[$entity_type][$index][$relationship_id] = $entity_id;
          }
        }
      }
    }

    // Load all entities and assign them to the correct result row.
    foreach ($entity_ids_by_type as $entity_type => $ids) {
      $entity_storage = $this->entityTypeManager->getStorage($entity_type);
      $flat_ids = iterator_to_array(new \RecursiveIteratorIterator(new \RecursiveArrayIterator($ids)), FALSE);

      $entities = $entity_storage->loadMultiple(array_unique($flat_ids));
      $results = $this->assignEntitiesToResult($ids, $entities, $results);
    }
  }

  /**
   * Sets entities onto the view result row objects.
   *
   * This method takes into account the relationship in which the entity was
   * needed in the first place.
   *
   * @param mixed[] $ids
   *   An array of identifiers (entity ID / revision ID).
   * @param \Drupal\Core\Entity\EntityInterface[] $entities
   *   An array of entities keyed by their identified (entity ID / revision ID).
   * @param \Drupal\views\ResultRow[] $results
   *   The entire views result.
   *
   * @return \Drupal\views\ResultRow[]
   *   The changed views results.
   */
  protected function assignEntitiesToResult(array $ids, array $entities, array $results) {
    foreach ($ids as $index => $relationships) {
      foreach ($relationships as $relationship_id => $id) {
        $entity = NULL;

        if (isset($entities[$id])) {
          $entity = $entities[$id];
        }

        if ($entity) {
          if ($relationship_id == 'none') {
            $results[$index]->_entity = $entity;
          }
          else {
            $results[$index]->_relationship_entities[$relationship_id] = $entity;
          }
        }
      }
    }

    return $results;
  }

  /**
   * Gets all the involved entities of the view.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   All entities.
   */
  protected function getAllEntities() {
    $entities = [];
    foreach ($this->view->result as $row) {
      if ($row->_entity) {
        $entities[] = $row->_entity;
      }
      foreach ($row->_relationship_entities as $entity) {
        $entities[] = $entity;
      }
    }

    return $entities;
  }

  /**
   * Returns total number of hits from the Elasticsearch result.
   *
   * @param array|object $data
   *   Data.
   *
   * @return int
   *   Hit count.
   */
  protected function getTotalHits($data) {
    if (ElasticsearchClientVersion::getMajorVersion() >= 7) {
      $total_items = $data['hits']['total']['value'] ?? 0;
    }
    else {
      $total_items = $data['hits']['total'] ?? 0;
    }

    return $total_items;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = [];

    foreach ($this->getAllEntities() as $entity) {
      $tags = Cache::mergeTags($entity->getCacheTags(), $tags);
    }

    foreach ($this->getEntityRelationships() as $relationship) {
      if (isset($relationship['entity_type_key'])) {
        $entity_type_id = $this->getNestedValue($relationship['entity_type_key'], []);

        if ($entity_type = $this->entityTypeManager->getDefinition($entity_type_id, FALSE)) {
          $tags = Cache::mergeTags($tags, $entity_type->getListCacheTags());
        }
      }
    }

    // Get cache tags from the query builder.
    if ($query_builder = $this->getQueryBuilder()) {
      $tags = Cache::mergeTags($tags, $query_builder->getCacheTags());
    }

    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    $contexts = [];

    // Get cache context from the query builder.
    if ($query_builder = $this->getQueryBuilder()) {
      $contexts = Cache::mergeContexts($contexts, $query_builder->getCacheContexts());
    }

    foreach ($this->getEntityRelationships() as $relationship) {
      if (isset($relationship['entity_type_key'])) {
        $entity_type_id = $this->getNestedValue($relationship['entity_type_key'], []);

        if ($entity_type = $this->entityTypeManager->getDefinition($entity_type_id, FALSE)) {
          $contexts = Cache::mergeContexts($contexts, $entity_type->getListCacheContexts());
        }
      }
    }

    return $contexts;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    $max_age = parent::getCacheMaxAge();
    foreach ($this->getAllEntities() as $entity) {
      $max_age = Cache::mergeMaxAges($max_age, $entity->getCacheMaxAge());
    }

    return $max_age;
  }

}
