(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Utility: fetch text/json safely.
   */
  async function fetchText(url) {
    const res = await fetch(url, { credentials: 'same-origin' });
    if (!res.ok) throw new Error(`Failed to load ${url}: ${res.status}`);
    return await res.text();
  }
  async function fetchJson(url) {
    const res = await fetch(url, { credentials: 'same-origin' });
    if (!res.ok) throw new Error(`Failed to load ${url}: ${res.status}`);
    return await res.json();
  }

  /**
   * Utility: show Reveal-* elements on Marker-* hover.
   */
  function suffixAfterFirstDash(id = '') {
    const i = id.indexOf('-');
    return i === -1 ? '' : id.slice(i + 1); // X may contain hyphens; we keep them
  }
  function findRevealFor(svgRoot, id) {
    // Only Marker-* is allowed to reveal.
    if (!/^marker-/i.test(id)) return null;

    const x = suffixAfterFirstDash(id);
    if (!x) return null;
    return svgRoot.querySelector(`[id="Reveal-${CSS.escape(x)}"]`);
  }

  /**
   * Utility: fetch text/json safely.
   */
  const DPR = window.devicePixelRatio || 1;
  function snapToDevicePixels(n) {
    return Math.round(n * DPR) / DPR;
  }

  /**
   * Position a popover next to an SVG element (using screen coords).
   */
  function positionPopover(popoverEl, svgEl, containerEl, opts = {}) {
    const { gap = 8 } = opts; // spacing between marker and popover
    const r = svgEl.getBoundingClientRect();
    const c = containerEl.getBoundingClientRect();

    const markerCenterX = r.left + (r.width / 2);

    // Measure popover after it’s visible so offsetWidth/Height are real.
    const pw = popoverEl.offsetWidth;
    const ph = popoverEl.offsetHeight;

    // Default placement: directly under the marker, horizontally centered.
    let top  = (r.bottom - c.top) + gap; 
    let left = (markerCenterX - c.left) - (pw / 2);

    // Keep it within the container horizontally.
    left = Math.max(8, Math.min(left, c.width - pw - 8));

    // If it overflows bottom, flip to top.
    if (top + ph > c.height) {
      top = (r.top - c.top) - ph - gap;
      if (top < 0) top = Math.max(8, c.height - ph - 8);
    }

    // Snap to whole device pixels to avoid blurry text.
    left = snapToDevicePixels(left);
    top  = snapToDevicePixels(top);

    popoverEl.style.transform = `translate(${left}px, ${top}px)`;
  }

  /**
   * Build popover content with nav controls.
   */
  function renderPopover(popoverEl, data, markerId, order, opts = {}) {
    const enableNav = opts.enableNav !== false;
    const idx = order.indexOf(markerId);
    const prevId = order[(idx - 1 + order.length) % order.length];
    const nextId = order[(idx + 1) % order.length];

    const item = data.items[markerId] || { title: markerId, html: '<p>No data.</p>' };

    popoverEl.innerHTML = `
      <div class="isv-popover__header">
        <div class="isv-popover__title">${item.title || markerId}</div>
        <button class="isv-popover__close" aria-label="Close">&times;</button>
      </div>
      <div class="isv-popover__body">${item.html || ''}</div>
      ${enableNav ? `
        <div class="isv-popover__footer">
          <button class="isv-popover__nav" data-target="${prevId}" aria-label="Previous">&#171; Prev</button>
          <button class="isv-popover__nav" data-target="${nextId}" aria-label="Next">Next &#187;</button>
        </div>` : ``}
    `;
  }

  /**
   * Main initializer per container.
   */
  async function init(containerId, settings) {
    const root = document.getElementById(containerId);
    const STR_CLICK_TO_OPEN = Drupal.t('Click to open', {}, { context: 'interactive_svg' });
    if (!root) return;

    const stage = root.querySelector('.interactive-svg-map__stage');
    const pop = root.querySelector('.interactive-svg-map__popover');

    // Bind once per container to keep it simple; harmless if multiple maps exist.
    if (!root.dataset.isvEscBound) {
      document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
          // Closes this container's popover if open.
          closePopover();
        }
      }, true);
      root.dataset.isvEscBound = '1';
    }

    // Close on click outside of popover.
    function onDocPointerDown(e) {
      if (pop.hidden) return;

      // Use composedPath() to be robust across Shadow DOM.
      const path = typeof e.composedPath === 'function' ? e.composedPath() : [];
      const t = e.target;

      const insidePop =
        pop.contains(t) || (path.length && path.includes(pop));

      // Clicking a marker should not close it.
      const insideMarker =
        svg.contains(t) && !!t.closest(`${markerSelector}`);

      if (!insidePop && !insideMarker) {
        closePopover();
      }
    }

    // Bind once per container.
    if (!root.dataset.isvOutsideBound) {
      // pointerdown catches both mouse and touch, fires before click.
      document.addEventListener('pointerdown', onDocPointerDown, true); // capture
      root.dataset.isvOutsideBound = '1';
    }

    if (!settings.svgUrl || !settings.jsonUrl) {
      stage.innerHTML = '<p>' + Drupal.t('Interactive map is not configured.') + '</p>';
      return;
    }

    let activeMarkerId = null;
    function isPopoverOpen() { return !pop.hidden; }

    function setActiveMarker(newId) {
      // Clear previous
      if (activeMarkerId) {
        const prev = svg.querySelector(`[id="${CSS.escape(activeMarkerId)}"]`);
        if (prev) {
          prev.classList.remove('isv-marker--active');
          prev.setAttribute('aria-expanded', 'false');
        }
      }

      activeMarkerId = newId || null;

      // Set new
      if (activeMarkerId) {
        const next = svg.querySelector(`[id="${CSS.escape(activeMarkerId)}"]`);
        if (next) {
          next.classList.add('isv-marker--active');
          next.setAttribute('aria-expanded', 'true');

          // Keep Reveal-* in sync with the “active” one.
          const r = findRevealFor(svg, activeMarkerId);
          if (r) showReveal(r);
        }
      }
    }

    function getActiveReveal() {
      return activeMarkerId ? findRevealFor(svg, activeMarkerId) : null;
    }

    function closePopover() {
      if (pop.hidden) return;
      pop.hidden = true;
      setActiveMarker(null);
      hideRevealLater(currentReveal, 0);
      deactivateFocusTrap();
    }

    let lastFocusBeforeOpen = null;

    function getFocusable(container) {
      return Array.from(container.querySelectorAll(
        'a[href], button:not([disabled]), textarea:not([disabled]), input:not([disabled]), select:not([disabled]), [tabindex]:not([tabindex="-1"])'
      )).filter(el => el.offsetParent !== null || el === container); // visible-ish
    }

    function focusFirstIn(container) {
      const f = getFocusable(container);
      (f[0] || container).focus();
    }

    function activateFocusTrap() {
      // Remember where we came from
      if (document.activeElement && document.activeElement !== document.body) {
        lastFocusBeforeOpen = document.activeElement;
      }
      // Initial focus into the popup
      focusFirstIn(pop);

      // Trap TAB/SHIFT+TAB within popup
      pop.addEventListener('keydown', trapKeydown, true);
      document.addEventListener('keydown', globalTrapKeydown, true);
    }

    function deactivateFocusTrap() {
      pop.removeEventListener('keydown', trapKeydown, true);
      document.removeEventListener('keydown', globalTrapKeydown, true);

      // Restore focus to prior element (e.g., marker) if still in DOM
      if (lastFocusBeforeOpen && document.contains(lastFocusBeforeOpen)) {
        lastFocusBeforeOpen.focus();
      }
      lastFocusBeforeOpen = null;
    }

    function trapKeydown(e) {
      if (e.key !== 'Tab') return;
      const canFocus = getFocusable(pop);
      if (canFocus.length === 0) {
        e.preventDefault();
        pop.focus();
        return;
      }
      const first = canFocus[0];
      const last  = canFocus[canFocus.length - 1];
      const isShift = e.shiftKey;

      if (!isShift && document.activeElement === last) {
        e.preventDefault();
        first.focus();
      } else if (isShift && document.activeElement === first) {
        e.preventDefault();
        last.focus();
      }
    }

    // If focus somehow escapes (e.g., via browser chrome), pull it back while open.
    function globalTrapKeydown(e) {
      if (e.key !== 'Tab') return;
      if (!isPopoverOpen()) return;
      if (!pop.contains(document.activeElement)) {
        e.preventDefault();
        focusFirstIn(pop);
      }
    }

    // Load SVG as text and import inline.
    const svgText = await fetchText(settings.svgUrl);
    const parser = new DOMParser();
    const doc = parser.parseFromString(svgText, 'image/svg+xml');
    const svg = doc.documentElement;

    // Ensure viewBox exists so zoom/fit works.
    if (!svg.getAttribute('viewBox')) {
      const w = svg.getAttribute('width') || 1000;
      const h = svg.getAttribute('height') || 800;
      svg.setAttribute('viewBox', `0 0 ${parseFloat(w)} ${parseFloat(h)}`);
    }

    // Inject SVG into the page.
    stage.innerHTML = '';
    stage.appendChild(svg);

    // Keep only one Reveal-* visible at a time and avoid flicker when moving
    let currentReveal = null;
    let hideTimer = null;

    function showReveal(reveal) {
      if (!reveal) return;

      // If a popup is open for some marker, lock the Reveal to that marker.
      const activeReveal = getActiveReveal();
      if (isPopoverOpen() && activeReveal && reveal !== activeReveal) {
        // Ignore attempts to switch Reveal due to hover on other markers.
        return;
      }

      if (currentReveal && currentReveal !== reveal) {
        currentReveal.classList.remove('isv-reveal--visible');
      }
      reveal.classList.add('isv-reveal--visible');
      currentReveal = reveal;
    }
    function hideRevealLater(reveal, delay = 120) {
      clearTimeout(hideTimer);

      // If this reveal is the active one and a popup is open, never hide it.
      const activeReveal = getActiveReveal();
      if (isPopoverOpen() && activeReveal && reveal === activeReveal) return;

      hideTimer = setTimeout(() => {
        // Re-check inside the timer for safety.
        const stillActiveReveal = getActiveReveal();
        if (isPopoverOpen() && stillActiveReveal && reveal === stillActiveReveal) return;

        if (reveal && reveal === currentReveal) {
          reveal.classList.remove('isv-reveal--visible');
          currentReveal = null;
        }
      }, delay);
    }

    /**
     * Ensure the viewBox covers the full SVG.
     * Use getBBox() AFTER the SVG is in the DOM.
     */
    try {
      // { x, y, width, height }
      const bbox = svg.getBBox();
      // If bbox is valid, set/overwrite viewBox to match actual drawing extents.
      if (bbox && bbox.width && bbox.height) {
        svg.setAttribute('viewBox', `${bbox.x} ${bbox.y} ${bbox.width} ${bbox.height}`);
      } else if (!svg.hasAttribute('viewBox')) {
        // Fallback if bbox fails: derive from width/height.
        const w = parseFloat(svg.getAttribute('width')) || 1000;
        const h = parseFloat(svg.getAttribute('height')) || 800;
        svg.setAttribute('viewBox', `0 0 ${w} ${h}`);
      }
    } catch (e) {
      // If getBBox() fails (for example, when the SVG isn't yet rendered or is hidden),
      // fall back to using the SVG's width and height attributes to define the viewBox.
      if (!svg.hasAttribute('viewBox')) {
        const w = parseFloat(svg.getAttribute('width')) || 1000;
        const h = parseFloat(svg.getAttribute('height')) || 800;
        svg.setAttribute('viewBox', `0 0 ${w} ${h}`);
      }
    }

    // Fit entire SVG inside the box (keeps aspect, no crop).
    svg.setAttribute('preserveAspectRatio', 'xMidYMid meet');

    // Let CSS/container drive size.
    svg.style.width = '100%';
    svg.style.height = '100%';
    svg.style.display = 'block';
    svg.style.overflow = 'visible';

    // Load JSON data.
    const data = await fetchJson(settings.jsonUrl);
    const ITEMS = (data && data.items) || {};
    const hasContentFor = (id) => {
      const it = ITEMS[id];
      return !!(it && (
        (typeof it.html === 'string' && it.html.trim()) ||
        (typeof it.title === 'string' && it.title.trim())
      ));
    };

    // Turn any string/HTML into plain text.
    function toPlainText(s) {
      if (!s) return '';
      const tmp = document.createElement('div');
      tmp.innerHTML = s;
      return tmp.textContent.trim();
    }

    // Build a readable label for a marker.
    function getMarkerAriaLabel(id) {
      const it = ITEMS[id];
      if (!it) return id;
      if (it.title && it.title.trim()) return toPlainText(it.title);
      if (it.html && it.html.trim()) {
        const txt = toPlainText(it.html);
      }
      return id;
    }

    // Add a title element as a tooltip.
    function setSvgTooltip(node, text) {
      // Remove existing direct <title> children to avoid duplicates.
      node.querySelectorAll(':scope > title').forEach(t => t.remove());
      const t = document.createElementNS('http://www.w3.org/2000/svg', 'title');
      t.textContent = text || '';
      // Place it as first child.
      node.insertBefore(t, node.firstChild);
    }

    const order = Array.isArray(data.order) ? data.order : Object.keys(data.items || {});
    const markerSelector = settings.markerSelector || 'g[id^="marker-" i]';

    // Hook up SVG.js for animations and interactions.
    // SVG() comes from svg.min.js.
    const draw = SVG(svg);

    // Add hover animations and click handlers to markers.
    const markers = svg.querySelectorAll(markerSelector);

    markers.forEach((node) => {
      const id = node.getAttribute('id');
      if (!id) return;
      const reveal = findRevealFor(svg, id);

      const el = SVG(node);

      // Make markers focusable for a11y.
      node.setAttribute('tabindex', '0');
      node.setAttribute('role', 'button');
      node.setAttribute('aria-label', getMarkerAriaLabel(id));
      node.setAttribute('aria-expanded', 'false');
      node.classList.add('isv-marker');

      // Add tooltip text if marker has popup.
      if (hasContentFor(id)) {
        setSvgTooltip(node, STR_CLICK_TO_OPEN);
      }

      // Hover/focus: show the matching Reveal-X
      const onEnter = () => {
        node.classList.add('isv-marker--hover');

        // If a popup is open for a different marker, don't switch the reveal.
        if (isPopoverOpen() && activeMarkerId && activeMarkerId !== id) return;

        showReveal(reveal);
      };
      const onLeave = () => {
        node.classList.remove('isv-marker--hover');

        // If a popup is open for a marker, don't hide anything.
        if (isPopoverOpen() && activeMarkerId) return;

        // Defer hide so pointer can move onto the reveal without flicker
        hideRevealLater(reveal);
      };

      // Hover/focus state via CSS.
      node.addEventListener('mouseenter', onEnter);
      node.addEventListener('mouseleave', onLeave);
      node.addEventListener('focus',     onEnter);
      node.addEventListener('blur',      onLeave);

      // Click/Enter to open popover.
      function openPopover() {
        // If this marker isn't defined in JSON, do nothing.
        if (!hasContentFor(id)) return;

        setActiveMarker(id);

        renderPopover(pop, data, id, order, { enableNav: settings.enableNav !== false });
        pop.hidden = false;

        // Save current target so we can re-position on resize/scroll.
        pop.dataset.currentTargetId = id;

        // Two RAFs ensure the browser has laid out the new content (sizes are accurate).
        requestAnimationFrame(() => {
          requestAnimationFrame(() => {
            positionPopover(pop, node, root, { gap: 8 });
            activateFocusTrap();
          });
        });
      }

      // Make popover focusable.
      pop.setAttribute('role', 'dialog');
      pop.setAttribute('aria-modal', 'true');
      pop.setAttribute('tabindex', '-1');

      // Delegate all clicks inside the popover.
      pop.addEventListener('click', (e) => {
        // Close
        const closeBtn = e.target.closest('.isv-popover__close');
        if (closeBtn) {
          closePopover();
          return;
        }

        // Prev/Next buttons.
        const navBtn = e.target.closest('.isv-popover__nav');
        if (navBtn) {
          const target = navBtn.getAttribute('data-target');
          const targetNode = svg.querySelector(`[id="${CSS.escape(target)}"]`);
          if (targetNode) {
            setActiveMarker(target);
            renderPopover(pop, data, target, order, { enableNav: settings.enableNav !== false });
            pop.dataset.currentTargetId = target;
            // Wait for layout, then reposition under the new marker.
            requestAnimationFrame(() => {
              requestAnimationFrame(() => {
                positionPopover(pop, targetNode, root, { gap: 8 });
                focusFirstIn(pop);
              });
            });
            // keep keyboard focus inside the popover.
            pop.focus();
          }
        }
      });

      pop.addEventListener('mouseenter', () => clearTimeout(hideTimer));
      pop.addEventListener('mouseleave', () => {
        // Only hide if nothing is active
        if (!activeMarkerId) hideRevealLater(currentReveal, 80);
      });
      pop.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
          closePopover();
        }
      });

      function repositionActivePopover() {
        if (pop.hidden) return;
        const targetId = pop.dataset.currentTargetId;
        if (!targetId) return;
        const targetNode = svg.querySelector(`[id="${CSS.escape(targetId)}"]`);
        if (targetNode) positionPopover(pop, targetNode, root, { gap: 8 });
      }

      window.addEventListener('resize', repositionActivePopover);
      window.addEventListener('scroll', repositionActivePopover, { passive: true });

      // Visually mark undefined markers as non-interactive.
      if (!hasContentFor(id)) {
        node.dataset.disabled = 'true';
        node.style.cursor = 'default';
      }
      node.addEventListener('click', openPopover);

      node.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          openPopover();
        }
      });
    });

    // Resize/scroll: keep popover near marker.
    window.addEventListener('resize', () => {
      if (!pop.hidden) {
        // Find current target by title text in footer buttons.
        const current = pop.querySelector('.isv-popover__title')?.textContent;
      }
    });
  }

  Drupal.behaviors.interactiveSvgMap = {
    attach: function (context) {
      const cfg = drupalSettings.interactiveSvgMap || {};
      Object.keys(cfg).forEach((containerId) => {
        const el = context.querySelector('#' + containerId);
        if (el && !el.dataset.isvInitialized) {
          el.dataset.isvInitialized = '1';
          init(containerId, cfg[containerId]).catch((e) => {
            console.error(e);
            const stage = el.querySelector('.interactive-svg-map__stage');
            if (stage) {
              stage.innerHTML = '<p>' + Drupal.t('There was a problem loading the interactive map.') + '</p>';
            }
          });
        }
      });
    }
  };

})(Drupal, drupalSettings);
