<?php

namespace Drupal\interactive_svg\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\file\Entity\File;
use Drupal\file\FileUsage\FileUsageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an Interactive SVG block.
 *
 * @Block(
 *   id = "interactive_svg_block",
 *   admin_label = @Translation("Interactive SVG")
 * )
 */
class InteractiveSvgBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Generates public/absolute URLs for file URIs.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * Tracks usage of files by this block instance.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected FileUsageInterface $fileUsage;

  /**
   * UUID service for per-instance usage IDs.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected UuidInterface $uuid;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->fileUrlGenerator = $container->get('file_url_generator');
    $instance->fileUsage = $container->get('file.usage');
    $instance->uuid = $container->get('uuid');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'svg_fid' => NULL,
      'json_fid' => NULL,
      'container_height' => '600px',
      'marker_selector' => 'g[id^="marker-"]',
      'enable_nav' => TRUE,
      // Unique per placed-block, used for file usage tracking & DOM id.
      'usage_id' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form['svg_fid'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('SVG file'),
      '#upload_location' => 'public://interactive_svg',
      '#upload_validators' => ['file_validate_extensions' => ['svg']],
      '#default_value' => $this->configuration['svg_fid'] ? [$this->configuration['svg_fid']] : NULL,
      '#required' => TRUE,
      '#description' => $this->t('Upload an SVG. Clickable markers should be <code>&lt;g id="marker-..."&gt;</code> (or change the selector below).'),
    ];

    $form['json_fid'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('JSON data file'),
      '#upload_location' => 'public://interactive_svg',
      '#upload_validators' => ['file_validate_extensions' => ['json']],
      '#default_value' => $this->configuration['json_fid'] ? [$this->configuration['json_fid']] : NULL,
      '#required' => TRUE,
      '#description' => $this->t('Upload JSON mapping marker IDs to popover content.'),
    ];

    $form['container_height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Container max-height'),
      '#default_value' => $this->configuration['container_height'],
      '#description' => $this->t('Any valid CSS max-height (e.g., 600px).'),
    ];

    $form['marker_selector'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Marker CSS selector'),
      '#default_value' => $this->configuration['marker_selector'],
      '#description' => $this->t('Selector for clickable markers. Default: <code>g[id^="marker-"]</code>.'),
    ];

    $form['enable_nav'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable popover navigation (Prev/Next)'),
      '#default_value' => !isset($this->configuration['enable_nav']) ? TRUE : (bool) $this->configuration['enable_nav'],
      '#description' => $this->t('Uncheck to hide the Prev/Next buttons in the popover.'),
    ];

    $form['example'] = [
      '#type' => 'details',
      '#title' => $this->t('Example JSON'),
      '#open' => FALSE,
      'text' => [
        '#markup' => '<pre>' . json_encode(self::exampleJson(), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . '</pre>',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    // Ensure a unique usage_id per placed block for file usage accounting.
    if (empty($this->configuration['usage_id'])) {
      $this->configuration['usage_id'] = $this->uuid->generate();
    }
    $usage_id = $this->configuration['usage_id'];

    foreach (['svg_fid', 'json_fid'] as $key) {
      $fid = $form_state->getValue($key);
      if (!empty($fid[0]) && ($file = File::load($fid[0]))) {
        // Make files permanent and record usage against this
        // specific block instance.
        $file->setPermanent();
        $file->save();
        $this->fileUsage->add($file, 'interactive_svg', 'block', $usage_id);
        $this->configuration[$key] = $file->id();
      }
    }

    $this->configuration['container_height'] = $form_state->getValue('container_height');
    $this->configuration['marker_selector'] = $form_state->getValue('marker_selector');
    $this->configuration['enable_nav'] = (bool) $form_state->getValue('enable_nav');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [
      '#attached' => [
        'library' => ['interactive_svg/svg'],
      ],
    ];

    // Resolve file URLs.
    $svg_url = $json_url = NULL;
    $cache_tags = $build['#cache']['tags'] ?? [];

    if (!empty($this->configuration['svg_fid']) && ($f = File::load($this->configuration['svg_fid']))) {
      $svg_url = $this->fileUrlGenerator->generateString($f->getFileUri());
      $cache_tags = Cache::mergeTags($cache_tags, $f->getCacheTags());
    }
    if (!empty($this->configuration['json_fid']) && ($f = File::load($this->configuration['json_fid']))) {
      $json_url = $this->fileUrlGenerator->generateString($f->getFileUri());
      $cache_tags = Cache::mergeTags($cache_tags, $f->getCacheTags());
    }

    // Unique container ID per instance to allow multiple blocks on a page.
    $usage_id = $this->configuration['usage_id'] ?? $this->uuid->generate();
    $container_id = 'interactive-svg-' . substr(hash('sha256', $usage_id), 0, 8) . '-' . uniqid();
    $height = $this->configuration['container_height'] ?? '600px';

    $build['container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => $container_id,
        'class' => ['interactive-svg-map'],
        'style' => 'position:relative; max-height:' . $height . ';',
      ],
      'svg_holder' => [
        '#markup' => '<div class="interactive-svg-map__stage" aria-live="polite"></div>'
        . '<div class="interactive-svg-map__popover" hidden></div>',
      ],
    ];

    // Pass per-instance settings to JS.
    $build['#attached']['drupalSettings']['interactiveSvgMap'][$container_id] = [
      'svgUrl' => $svg_url,
      'jsonUrl' => $json_url,
      'markerSelector' => $this->configuration['marker_selector'] ?? 'g[id^="marker-"]',
      'enableNav' => isset($this->configuration['enable_nav']) ? (bool) $this->configuration['enable_nav'] : TRUE,
    ];

    // Cacheability: vary by URL and permissions, invalidate on file change.
    $build['#cache']['contexts'] = ['url', 'user.permissions'];
    $build['#cache']['tags'] = $cache_tags;
    // @todo Keep conservative during development (increase later).
    $build['#cache']['max-age'] = 0;

    return $build;
  }

  /**
   * Example JSON payload shown in the form.
   */
  public static function exampleJson(): array {
    return [
      'order' => [
        'Marker-1',
        'Marker-2',
        'Marker-3',
      ],
      'items' => [
        'Marker-1' => [
          'title' => 'First Location',
          'html' => '<p>This is the <strong>first</strong> marker.</p>',
        ],
        'Marker-2' => [
          'title' => 'Second Location',
          'html' => '<p>Details for marker 2. You can include lists, links, etc.</p>',
        ],
        'Marker-3' => [
          'title' => 'Third Location',
          'html' => '<p>Any HTML you need goes here.</p>',
        ],
      ],
    ];
  }

}
