<?php

declare(strict_types=1);

namespace Drupal\interface_translation_auto;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;

/**
 * Provides a service for interacting with the DeepL Translation API.
 *
 * This service handles all communication with the DeepL API, including
 * authentication, translation requests, and usage information retrieval.
 */
class DeepLClient {

  use StringTranslationTrait;

  /**
   * The API version.
   *
   * @var string
   */
  protected const API_VERSION = 'v2';

  /**
   * The base URLs for different API types.
   *
   * @var string[]
   */
  protected const API_BASE_URLS = [
    'free' => 'https://api-free.deepl.com/' . self::API_VERSION,
    'pro' => 'https://api.deepl.com/' . self::API_VERSION,
  ];

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $logger;

  /**
   * Constructs a new DeepLClient.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \GuzzleHttp\ClientInterface $client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $client,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->configFactory = $config_factory;
    $this->client = $client;
    $this->logger = $logger_factory->get('interface_translation_auto');
  }

  /**
   * Gets the API endpoint based on type.
   *
   * @param string $api_type
   *   The API type (free or pro).
   *
   * @return string
   *   The complete API endpoint URL.
   */
  protected function getApiEndpoint(string $api_type = 'free'): string {
    return static::API_BASE_URLS[$api_type];
  }

  /**
   * Tests the DeepL API connection.
   *
   * Validates the API key and connection by making a test request
   * to the usage endpoint.
   *
   * @param string $api_key
   *   The API key to test.
   * @param string $api_type
   *   The type of API (free/pro).
   *
   * @return bool
   *   TRUE if connection is successful, FALSE otherwise.
   */
  public function testConnection(string $api_key, string $api_type = 'free'): bool {
    try {
      if (empty($api_key)) {
        throw new \Exception('API key is required.');
      }

      $request = new Request(
        'GET',
        $this->getApiEndpoint($api_type) . '/usage',
        [
          'Authorization' => 'DeepL-Auth-Key ' . $api_key,
          'Content-Type' => 'application/json',
        ]
      );

      $response = $this->client->send($request);
      $result = json_decode($response->getBody()->getContents(), TRUE);

      // Check if the response contains usage information.
      if (isset($result['character_count']) || isset($result['character_limit'])) {
        $this->logger->notice('DeepL API connection test successful for @type API.', [
          '@type' => $api_type,
        ]);
        return TRUE;
      }

      return FALSE;
    }
    catch (GuzzleException $e) {
      $this->logger->error('DeepL API connection test failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('DeepL API connection test error: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Translates text using DeepL API.
   *
   * @param string $text
   *   Text to translate.
   * @param string $target_lang
   *   Target language code.
   *
   * @return string|null
   *   Translated text or NULL if translation failed.
   *
   * @throws \Exception
   *   Throws exception when translation fails.
   */
  public function translate(string $text, string $target_lang): ?string {
    if (empty($text)) {
      throw new \Exception('Empty text provided for translation.');
    }

    if (empty($target_lang)) {
      throw new \Exception('No target language specified.');
    }

    try {
      $config = $this->configFactory->get('interface_translation_auto.settings');
      $api_key = $config->get('api_key');
      $api_type = $config->get('api_type') ?? 'free';

      if (empty($api_key)) {
        throw new \Exception('DeepL API key is not configured.');
      }

      $this->logger->notice('Attempting to translate text to @lang: @text', [
        '@lang' => $target_lang,
        '@text' => $text,
      ]);

      $request = new Request(
        'POST',
        $this->getApiEndpoint($api_type) . '/translate',
        [
          'Authorization' => 'DeepL-Auth-Key ' . $api_key,
          'Content-Type' => 'application/json',
        ]
      );

      $response = $this->client->send($request, [
        'json' => [
          'text' => [$text],
          'target_lang' => strtoupper($target_lang),
        ],
      ]);

      $result = json_decode($response->getBody()->getContents(), TRUE);

      if (empty($result['translations'][0]['text'])) {
        throw new \Exception('No translation found in DeepL response.');
      }

      return $result['translations'][0]['text'];
    }
    catch (GuzzleException $e) {
      $this->logger->error('DeepL API request failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      throw new \Exception('DeepL API request failed: ' . $e->getMessage());
    }
    catch (\Exception $e) {
      $this->logger->error('Translation error: @error', [
        '@error' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Retrieves usage information from the DeepL API.
   *
   * Gets information about the API usage including character count
   * and limits for the current billing period.
   *
   * @return array
   *   Array containing usage information.
   *
   * @throws \Exception
   *   Throws exception when unable to retrieve usage information.
   */
  public function getUsage(): array {
    try {
      $config = $this->configFactory->get('interface_translation_auto.settings');
      $api_key = $config->get('api_key');
      $api_type = $config->get('api_type') ?? 'free';

      if (empty($api_key)) {
        throw new \Exception('DeepL API key is not configured.');
      }

      $request = new Request(
        'GET',
        $this->getApiEndpoint($api_type) . '/usage',
        [
          'Authorization' => 'DeepL-Auth-Key ' . $api_key,
          'Content-Type' => 'application/json',
        ]
      );

      $response = $this->client->send($request);
      $result = json_decode($response->getBody()->getContents(), TRUE);

      if (!isset($result['character_count']) || !isset($result['character_limit'])) {
        throw new \Exception('Invalid usage response from DeepL API.');
      }

      return $result;
    }
    catch (GuzzleException $e) {
      $this->logger->error('DeepL API usage request failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      throw new \Exception('Failed to get DeepL API usage information.');
    }
  }

}
