<?php

declare(strict_types=1);

namespace Drupal\interface_translation_auto\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\interface_translation_auto\DeepLClient;
use Drupal\interface_translation_auto\OpenAiClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a configuration form for DeepL API settings.
 */
class InterfaceTranslationAutoSettingsForm extends ConfigFormBase {

  /**
   * The DeepL client service.
   *
   * @var \Drupal\interface_translation_auto\DeepLClient
   */
  protected $deeplClient;

  /**
   * The OpenAI client service.
   *
   * @var \Drupal\interface_translation_auto\OpenAiClient
   */
  protected $openaiClient;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'interface_translation_auto_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['interface_translation_auto.settings'];
  }

  /**
   * Constructs a new InterfaceTranslationAutoSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed configuration manager.
   * @param \Drupal\interface_translation_auto\DeepLClient $deepl_client
   *   The DeepL client service.
   * @param \Drupal\interface_translation_auto\OpenAiClient $openai_client
   *   The OpenAI client service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    DeepLClient $deepl_client,
    OpenAiClient $openai_client,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->deeplClient = $deepl_client;
    $this->openaiClient = $openai_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('interface_translation_auto.deepl_client'),
      $container->get('interface_translation_auto.openai_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('interface_translation_auto.settings');

    $form['provider'] = [
      '#type' => 'select',
      '#title' => $this->t('Translation Provider'),
      '#default_value' => $config->get('provider') ?: 'deepl',
      '#options' => [
        'deepl' => $this->t('DeepL'),
        'openai' => $this->t('OpenAI'),
      ],
      '#required' => TRUE,
    ];

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('DeepL API Key'),
      '#default_value' => $config->get('api_key'),
      '#required' => FALSE,
      '#description' => $this->t('Enter your DeepL API key. Free API keys have a ":fx" suffix.'),
      '#states' => [
        'visible' => [
          ':input[name="provider"]' => ['value' => 'deepl'],
        ],
      ],
    ];

    $form['api_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('API Type'),
      '#default_value' => $config->get('api_type') ?: 'free',
      '#options' => [
        'free' => $this->t('Free API (api-free.deepl.com)'),
        'pro' => $this->t('Pro API (api.deepl.com)'),
      ],
      '#required' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="provider"]' => ['value' => 'deepl'],
        ],
      ],
    ];

    $form['openai_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenAI API Key'),
      '#default_value' => $config->get('openai_api_key'),
      '#maxlength' => 512,
      '#required' => FALSE,
      '#description' => $this->t('Enter your OpenAI API key.'),
      '#states' => [
        'visible' => [
          ':input[name="provider"]' => ['value' => 'openai'],
        ],
      ],
    ];

    $form['openai_model'] = [
      '#type' => 'select',
      '#title' => $this->t('OpenAI Model'),
      '#default_value' => $config->get('openai_model') ?: 'gpt-5.2',
      '#options' => [
        'gpt-5.2' => $this->t('GPT-5.2'),
        'gpt-5.2-pro' => $this->t('GPT-5.2 pro'),
        'gpt-5-mini' => $this->t('GPT-5 mini'),
        'gpt-5-nano' => $this->t('GPT 5 nano'),
        'gpt-4.1-mini' => $this->t('GPT 4.1 mini'),
        'gpt-4.1-nano' => $this->t('GPT 4.1 nano'),

      ],
      '#required' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="provider"]' => ['value' => 'openai'],
        ],
      ],
    ];

    $form['test_connection_wrapper'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'test-connection-wrapper',
      ],
    ];

    $form['test_connection_wrapper']['test_connection'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#name' => 'test_connection',
      '#ajax' => [
        'callback' => '::testConnectionAjax',
        'wrapper' => 'connection-status',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Testing connection...'),
        ],
      ],
    ];

    $form['test_connection_wrapper']['connection_status'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'connection-status',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Handles AJAX callback for testing the DeepL API connection.
   */
  public function testConnectionAjax(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    $provider = $form_state->getValue('provider');

    try {
      if ($provider === 'openai') {
        $api_key = $form_state->getValue('openai_api_key');
        $model = $form_state->getValue('openai_model');
        $success = $this->openaiClient->testConnection($api_key, $model);
      }
      else {
        $api_key = $form_state->getValue('api_key');
        $api_type = $form_state->getValue('api_type');
        $success = $this->deeplClient->testConnection($api_key, $api_type);
      }

      if ($success) {
        $message = $this->t('Connection successful!');
        $class = 'messages messages--status';
      }
      else {
        $message = $this->t('Connection failed. Please check your API key and settings.');
        $class = 'messages messages--error';
      }
    }
    catch (\Exception $e) {
      $message = $this->t('Connection error: @error', ['@error' => $e->getMessage()]);
      $class = 'messages messages--error';
    }

    $response->addCommand(
      new HtmlCommand(
        '#connection-status',
        ['#markup' => "<div class=\"$class\">$message</div>"]
      )
    );

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getTriggeringElement()['#name'] !== 'test_connection') {
      parent::validateForm($form, $form_state);

      $provider = $form_state->getValue('provider');

      if ($provider === 'openai') {
        $api_key = $form_state->getValue('openai_api_key');
        $model = $form_state->getValue('openai_model');

        if (empty($api_key)) {
          $form_state->setError($form['openai_api_key'], $this->t('OpenAI API key is required.'));
        }
        if (empty($model)) {
          $form_state->setError($form['openai_model'], $this->t('OpenAI model is required.'));
        }

        if (!$form_state->hasAnyErrors()) {
          try {
            $success = $this->openaiClient->testConnection($api_key, $model);
            if (!$success) {
              $form_state->setError($form['openai_api_key'], $this->t('Connection test failed. Please verify your API key and model.'));
            }
          }
          catch (\Exception $e) {
            $form_state->setError($form['openai_api_key'], $this->t('Connection error: @error', ['@error' => $e->getMessage()]));
          }
        }
      }
      else {
        $api_key = $form_state->getValue('api_key');
        $api_type = $form_state->getValue('api_type');

        if (empty($api_key)) {
          $form_state->setError($form['api_key'], $this->t('DeepL API key is required.'));
        }
        if (empty($api_type)) {
          $form_state->setError($form['api_type'], $this->t('API type is required.'));
        }

        $is_free_key = str_ends_with($api_key, ':fx');
        if ($api_type === 'free' && !$is_free_key) {
          $form_state->setError($form['api_key'], $this->t('This appears to be a Pro API key. Please select "Pro API" as the API type.'));
        }
        elseif ($api_type === 'pro' && $is_free_key) {
          $form_state->setError($form['api_key'], $this->t('This appears to be a Free API key. Please select "Free API" as the API type.'));
        }

        if (!$form_state->hasAnyErrors()) {
          try {
            $success = $this->deeplClient->testConnection($api_key, $api_type);
            if (!$success) {
              $form_state->setError($form['api_key'], $this->t('Connection test failed. Please verify your API key and settings.'));
            }
          }
          catch (\Exception $e) {
            $form_state->setError($form['api_key'], $this->t('Connection error: @error', ['@error' => $e->getMessage()]));
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('interface_translation_auto.settings')
      ->set('provider', $form_state->getValue('provider'))
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('api_type', $form_state->getValue('api_type'))
      ->set('openai_api_key', $form_state->getValue('openai_api_key'))
      ->set('openai_model', $form_state->getValue('openai_model'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
