<?php

declare(strict_types=1);

namespace Drupal\interface_translation_auto;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;

/**
 * Provides a service for translating text with the OpenAI Responses API.
 */
class OpenAiClient {

  use StringTranslationTrait;

  /**
   * The OpenAI Responses API endpoint.
   *
   * @var string
   */
  protected const RESPONSES_ENDPOINT = 'https://api.openai.com/v1/responses';

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new OpenAiClient.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \GuzzleHttp\ClientInterface $client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $client,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->configFactory = $config_factory;
    $this->client = $client;
    $this->logger = $logger_factory->get('interface_translation_auto');
  }

  /**
   * Tests the OpenAI API connection using a small translation request.
   *
   * @param string $api_key
   *   The OpenAI API key.
   * @param string $model
   *   The OpenAI model identifier.
   *
   * @return bool
   *   TRUE if connection is successful, FALSE otherwise.
   */
  public function testConnection(string $api_key, string $model): bool {
    try {
      if (empty($api_key)) {
        throw new \Exception('API key is required.');
      }
      if (empty($model)) {
        throw new \Exception('Model is required.');
      }

      $result = $this->sendRequest($api_key, $model, $this->buildPrompt('English', 'Test'));
      if (!empty($result['status']) && $result['status'] == 'completed') {
        return TRUE;
      }
      else {
        return FALSE;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('OpenAI API connection test failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('OpenAI API connection test error: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Translates text using the OpenAI Responses API.
   *
   * @param string $text
   *   Text to translate.
   * @param string $target_lang
   *   Target language name or code.
   *
   * @return string|null
   *   Translated text or NULL if translation failed.
   *
   * @throws \Exception
   *   Throws exception when translation fails.
   */
  public function translate(string $text, string $target_lang): ?string {
    if (empty($text)) {
      throw new \Exception('Empty text provided for translation.');
    }
    if (empty($target_lang)) {
      throw new \Exception('No target language specified.');
    }

    $config = $this->configFactory->get('interface_translation_auto.settings');
    $api_key = $config->get('openai_api_key');
    $model = $config->get('openai_model');

    if (empty($api_key)) {
      throw new \Exception('OpenAI API key is not configured.');
    }
    if (empty($model)) {
      throw new \Exception('OpenAI model is not configured.');
    }

    try {
      $prompt = $this->buildPrompt($target_lang, $text);
      $result = $this->sendRequest($api_key, $model, $prompt);
      $output = $this->extractText($result);

      if ($output === NULL || $output === '') {
        throw new \Exception('No translation found in OpenAI response.');
      }

      return $output;
    }
    catch (GuzzleException $e) {
      $this->logger->error('OpenAI API request failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      throw new \Exception('OpenAI API request failed: ' . $e->getMessage());
    }
    catch (\Exception $e) {
      $this->logger->error('OpenAI translation error: @error', [
        '@error' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Builds the translation prompt.
   */
  protected function buildPrompt(string $target, string $text): string {
    $prompt = 'You are translating content related to Drupal CMS, and Drupal development. You will be provided the specific text to translate (content after "TRANSLATE:"). You will translate this into %target% and it should read naturally like fluent %target%. Do not translate HTML tags. (end of prompt)';
    $prompt = str_replace('%target%', $target, $prompt);

    return $prompt . "\nTRANSLATE:\n" . $text;
  }

  /**
   * Sends a request to the OpenAI Responses API.
   */
  protected function sendRequest(string $api_key, string $model, string $prompt): array {
    $request = new Request(
      'POST',
      static::RESPONSES_ENDPOINT,
      [
        'Authorization' => 'Bearer ' . $api_key,
        'Content-Type' => 'application/json',
      ]
    );

    $response = $this->client->send($request, [
      'json' => [
        'model' => $model,
        'input' => $prompt,
      ],
    ]);

    return json_decode($response->getBody()->getContents(), TRUE) ?? [];
  }

  /**
   * Extracts text from a Responses API payload.
   */
  protected function extractText(array $result): ?string {
    if (!empty($result['output_text']) && is_string($result['output_text'])) {
      return $result['output_text'];
    }

    if (!empty($result['output'][0]['content'][0]['text'])) {
      return $result['output'][0]['content'][0]['text'];
    }

    if (!empty($result['output'][0]['content'])) {
      foreach ($result['output'][0]['content'] as $content) {
        if (!empty($content['text'])) {
          return $content['text'];
        }
      }
    }

    return NULL;
  }

}
