<?php

declare(strict_types=1);

namespace Drupal\interface_translation_auto\Commands;

use Drupal\language\Entity\ConfigurableLanguage;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drush\Exceptions\CommandFailedException;

/**
 * Drush commands for interface translation auto.
 */
final class InterfaceTranslationAutoCommands extends DrushCommands {

  /**
   * Translates untranslated interface strings for a language.
   */
  #[CLI\Command(name: 'interface-translation-auto:translate', aliases: ['ita-translate'])]
  #[CLI\Argument(name: 'langcode', description: 'The language code to translate.')]
  #[CLI\Usage(name: 'drush interface-translation-auto:translate fr', description: 'Translate untranslated interface strings to French.')]
  #[CLI\ValidateModulesEnabled(modules: ['interface_translation_auto'])]
  public function translate(string $langcode): void {
    if ($langcode === 'en') {
      $this->logger()->warning(dt('English is the source language and does not require translation.'));
      return;
    }

    $language = ConfigurableLanguage::load($langcode);
    if (!$language) {
      throw new CommandFailedException(dt('Language code @langcode is not configured.', [
        '@langcode' => $langcode,
      ]));
    }

    if ($language->isLocked()) {
      throw new CommandFailedException(dt('Language code @langcode is locked and cannot be translated.', [
        '@langcode' => $langcode,
      ]));
    }

    if (!locale_is_translatable($langcode)) {
      throw new CommandFailedException(dt('Language code @langcode is not translatable.', [
        '@langcode' => $langcode,
      ]));
    }

    \Drupal::moduleHandler()->loadInclude('interface_translation_auto', 'module');
    interface_translation_auto_start_batch($langcode, $language->label());

    drush_backend_batch_process();
  }

}
