<?php

namespace Drupal\interstitial\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for the interstitial page.
 */
class InterstitialController extends ControllerBase {

  /**
   * Interstitial view.
   */
  public function view(Request $request): array {
    $config = $this->config('interstitial.settings');
    $raw_dest = $request->query->get('dest');

    if (!is_string($raw_dest) || $raw_dest === '') {
      return [
        '#type' => 'markup',
        '#markup' => $this->t('No destination provided.'),
      ];
    }

    // Accept only http, https, and internal pseudo-scheme.
    $allowed = ['http', 'https', 'internal'];
    $scheme = parse_url($raw_dest, PHP_URL_SCHEME);
    if (!$scheme || !in_array($scheme, $allowed, TRUE)) {
      return [
        '#type' => 'markup',
        '#markup' => $this->t('Invalid destination.'),
      ];
    }

    // Build destination URL and display string.
    if ($scheme === 'internal') {
      $path = preg_replace('/^internal:/', '', $raw_dest);
      if ($path === '' || $path[0] !== '/') {
        $path = '/' . ltrim((string) $path, '/');
      }
      $dest_url = Url::fromUserInput($path);
      $current = \Drupal::requestStack()->getCurrentRequest();
      $host = $current ? $current->getSchemeAndHttpHost() : '';
      $dest_display = $host . $path;
    }
    else {
      $dest_url = Url::fromUri($raw_dest);
      $dest_display = $raw_dest;
    }

    $site_name = (string) ($this->config('system.site')->get('name') ?? $this->t('this site'));
    $template = (string) ($config->get('message') ?? 'You are now leaving the site {site}. Click below to continue to {url}.');
    $message = strtr($template, [
      '{site}' => $site_name,
      '{url}' => $dest_display,
    ]);

    return [
      '#theme' => 'interstitial_page',
      '#message' => $message,
      '#destination_url' => $dest_url->toString(),
      '#cache' => ['max-age' => 0],
    ];
  }

}
