<?php

namespace Drupal\Tests\intl_date\Kernel;

use Drupal\Core\Form\FormState;
use Drupal\intl_date\IntlDate;
use Drupal\intl_date\IntlDateFormatListBuilder;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\node\Traits\NodeCreationTrait;

/**
 * Kernel level coverage for the intl_date module.
 *
 * @group intl_date
 */
class IntlDateKernelTest extends KernelTestBase {

  use NodeCreationTrait;

  /**
   * A reusable timestamp (2021-05-18 13:24:03 UTC).
   */
  private const TIMESTAMP = 1621322643;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'language',
    'locale',
    'field',
    'text',
    'node',
    'intl_date',
  ];

  /**
   * Tests formatting a date via a stored intl date format.
   */
  public function testFormatPatternUsesStoredConfig() {
    $storage = $this->container->get('entity_type.manager')->getStorage('intl_date_format');
    $date_format = $storage->create([
      'id' => 'custom_date',
      'label' => 'Custom date',
      'pattern' => 'yyyy/MM/dd',
    ]);
    $date_format->save();

    $this->assertSame('2021/05/18', IntlDate::formatPattern(self::TIMESTAMP, 'custom_date', 'en'));
  }

  /**
   * Ensures an exception is thrown when the format id does not exist.
   */
  public function testFormatPatternThrowsForMissingEntity() {
    $this->expectException(\IntlException::class);
    IntlDate::formatPattern(self::TIMESTAMP, 'missing_format', 'en');
  }

  /**
   * Verifies the list builder renders rows with a formatted preview.
   */
  public function testListBuilderRendersPatternPreview() {
    $storage = $this->container->get('entity_type.manager')->getStorage('intl_date_format');
    $date_format = $storage->create([
      'id' => 'list_test',
      'label' => 'List test format',
      'pattern' => 'dd MMM yyyy',
    ]);
    $date_format->save();

    $definition = $this->container->get('entity_type.manager')->getDefinition('intl_date_format');
    $list_builder = IntlDateFormatListBuilder::createInstance($this->container, $definition);

    $header = $list_builder->buildHeader();
    $this->assertArrayHasKey('label', $header);
    $this->assertArrayHasKey('pattern', $header);

    $row = $list_builder->buildRow($date_format);
    $this->assertSame('List test format', $row['label']);

    $expected = IntlDate::format($this->container->get('datetime.time')->getRequestTime(), 'dd MMM yyyy');
    $this->assertSame($expected, $row['pattern']);
  }

  /**
   * Tests IntlTimestampFormatter default settings.
   */
  public function testIntlTimestampFormatterDefaultSettings() {
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installConfig(['field', 'node']);

    // Create a date format.
    $storage = $this->container->get('entity_type.manager')->getStorage('intl_date_format');
    $date_format = $storage->create([
      'id' => 'medium',
      'label' => 'Medium',
      'pattern' => 'dd MMM yyyy',
    ]);
    $date_format->save();

    // Create a node type.
    $node_type = NodeType::create(['type' => 'article', 'name' => 'Article']);
    $node_type->save();

    // Create formatter instance.
    $formatter = $this->container->get('plugin.manager.field.formatter')->createInstance('intl_timestamp', [
      'field_definition' => $this->createMock('\Drupal\Core\Field\FieldDefinitionInterface'),
      'settings' => [],
      'label' => '',
      'view_mode' => 'default',
      'third_party_settings' => [],
    ]);

    $settings = $formatter->getSettings();
    $this->assertEquals('medium', $settings['date_format']);
    $this->assertEquals('', $settings['timezone']);
  }

  /**
   * Tests IntlTimestampFormatter viewElements method.
   */
  public function testIntlTimestampFormatterViewElements() {
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installConfig(['field', 'node']);
    $this->installSchema('system', ['sequences']);

    // Create a date format.
    $storage = $this->container->get('entity_type.manager')->getStorage('intl_date_format');
    $date_format = $storage->create([
      'id' => 'test_format',
      'label' => 'Test format',
      'pattern' => 'yyyy-MM-dd',
    ]);
    $date_format->save();

    // Create a node type.
    $node_type = NodeType::create(['type' => 'article', 'name' => 'Article']);
    $node_type->save();

    // Create a node with a created timestamp.
    $node = $this->createNode([
      'type' => 'article',
      'created' => self::TIMESTAMP,
    ]);

    // Get the field definition for the created field.
    $field_definition = $node->getFieldDefinition('created');

    // Create formatter instance.
    $formatter = $this->container->get('plugin.manager.field.formatter')->createInstance('intl_timestamp', [
      'field_definition' => $field_definition,
      'settings' => [
        'date_format' => 'test_format',
        'timezone' => '',
      ],
      'label' => '',
      'view_mode' => 'default',
      'third_party_settings' => [],
    ]);

    // Get the field items.
    $items = $node->get('created');

    // Render the field.
    $elements = $formatter->viewElements($items, 'en');

    $this->assertArrayHasKey(0, $elements);
    $this->assertEquals('2021-05-18', $elements[0]['#markup']);
    $this->assertArrayHasKey('#cache', $elements[0]);
    $this->assertContains('timezone', $elements[0]['#cache']['contexts']);
  }

  /**
   * Tests IntlTimestampFormatter settings summary.
   */
  public function testIntlTimestampFormatterSettingsSummary() {
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installConfig(['field', 'node']);

    // Create a date format.
    $storage = $this->container->get('entity_type.manager')->getStorage('intl_date_format');
    $date_format = $storage->create([
      'id' => 'custom',
      'label' => 'Custom',
      'pattern' => 'dd/MM/yyyy',
    ]);
    $date_format->save();

    // Create a node type.
    $node_type = NodeType::create(['type' => 'article', 'name' => 'Article']);
    $node_type->save();

    $node = $this->createNode(['type' => 'article']);
    $field_definition = $node->getFieldDefinition('created');

    // Test without timezone override.
    $formatter = $this->container->get('plugin.manager.field.formatter')->createInstance('intl_timestamp', [
      'field_definition' => $field_definition,
      'settings' => [
        'date_format' => 'custom',
        'timezone' => '',
      ],
      'label' => '',
      'view_mode' => 'default',
      'third_party_settings' => [],
    ]);

    $summary = $formatter->settingsSummary();
    $this->assertContains('Date format: custom', array_map('strval', $summary));

    // Test with timezone override.
    $formatter = $this->container->get('plugin.manager.field.formatter')->createInstance('intl_timestamp', [
      'field_definition' => $field_definition,
      'settings' => [
        'date_format' => 'custom',
        'timezone' => 'America/New_York',
      ],
      'label' => '',
      'view_mode' => 'default',
      'third_party_settings' => [],
    ]);

    $summary = $formatter->settingsSummary();
    $this->assertContains('Date format: custom', array_map('strval', $summary));
    $this->assertContains('Time zone: America/New_York', array_map('strval', $summary));
  }

  /**
   * Tests IntlTimestampFormatter settings form.
   */
  public function testIntlTimestampFormatterSettingsForm() {
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installConfig(['field', 'node']);

    // Create a date format.
    $storage = $this->container->get('entity_type.manager')->getStorage('intl_date_format');
    $date_format = $storage->create([
      'id' => 'short',
      'label' => 'Short',
      'pattern' => 'dd/MM',
    ]);
    $date_format->save();

    $date_format2 = $storage->create([
      'id' => 'long',
      'label' => 'Long',
      'pattern' => 'dd MMMM yyyy',
    ]);
    $date_format2->save();

    // Create a node type and node.
    $node_type = NodeType::create(['type' => 'article', 'name' => 'Article']);
    $node_type->save();
    $node = $this->createNode(['type' => 'article']);
    $field_definition = $node->getFieldDefinition('created');

    // Create formatter instance.
    $formatter = $this->container->get('plugin.manager.field.formatter')->createInstance('intl_timestamp', [
      'field_definition' => $field_definition,
      'settings' => [
        'date_format' => 'short',
        'timezone' => '',
      ],
      'label' => '',
      'view_mode' => 'default',
      'third_party_settings' => [],
    ]);

    $form = [];
    $form_state = new FormState();
    $settings_form = $formatter->settingsForm($form, $form_state);

    $this->assertArrayHasKey('date_format', $settings_form);
    $this->assertEquals('select', $settings_form['date_format']['#type']);
    $this->assertArrayHasKey('short', $settings_form['date_format']['#options']);
    $this->assertArrayHasKey('long', $settings_form['date_format']['#options']);
    $this->assertEquals('short', $settings_form['date_format']['#default_value']);

    $this->assertArrayHasKey('timezone', $settings_form);
    $this->assertEquals('select', $settings_form['timezone']['#type']);
    $this->assertEquals('', $settings_form['timezone']['#default_value']);
  }

}
