<?php

namespace Drupal\ip_anon\Drush\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\ip_anon\IpAnonymizeInterface;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;

/**
 * Provides Drush commands for Raven module.
 */
class IpAnonCommands extends DrushCommands {

  use AutowireTrait;

  final public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected DateFormatterInterface $dateFormatter,
    protected IpAnonymizeInterface $ipAnonymize,
  ) {
    parent::__construct();
  }

  /**
   * Scrub IP addresses from database tables.
   */
  #[CLI\Command(name: 'ip_anon:scrub')]
  #[CLI\Usage(name: 'drush ip_anon:scrub', description: 'If IP anonymization is enabled, scrub IP addresses from the database according to the configured retention period for each database table.')]
  public function scrub(): void {
    $logger = $this->logger();
    if ($this->configFactory->get('ip_anon.settings')->get('policy')) {
      $this->ipAnonymize->scrub();
      if ($logger) {
        $logger->success(dt('IP addresses scrubbed.'));
      }
    }
    else {
      if ($logger) {
        $logger->warning(dt('Retention policy is configured to preserve IP addresses.'));
      }
    }
  }

  /**
   * Display IP address retention policy.
   *
   * @phpstan-ignore missingType.iterableValue
   */
  #[CLI\Command(name: 'ip_anon:policy')]
  #[CLI\Usage(name: 'drush ip_anon:policy', description: 'Display IP address retention policy.')]
  public function policy(array $options = ['format' => 'table']): RowsOfFields {
    $logger = $this->logger();
    $config = $this->configFactory->get('ip_anon.settings');
    if (!$config->get('policy')) {
      if ($logger) {
        $logger->warning(dt('Retention policy is configured to preserve IP addresses.'));
      }
    }
    $rows = [];
    foreach ($this->ipAnonymize->getTables() as $table => $values) {
      $period = $config->get("period_$table");
      if (!is_numeric($period) || $period < 0) {
        $period = dt('Forever');
      }
      else {
        $period = $this->dateFormatter->formatInterval((int) $period);
      }
      $rows[] = ['table' => $table, 'retention period' => $period];
    }
    return new RowsOfFields($rows);
  }

}
