<?php

namespace Drupal\ip_anon\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ip_anon\IpAnonymizeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for ip_anon module.
 */
class IpAnonSettings extends ConfigFormBase {

  final public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected DateFormatterInterface $dateFormatter,
    protected IpAnonymizeInterface $ipAnonymize,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('date.formatter'),
      $container->get(IpAnonymizeInterface::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ip_anon_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @return string[]
   *   Editable config names.
   */
  protected function getEditableConfigNames() {
    return ['ip_anon.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @param mixed[] $form
   *   Settings form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return mixed[]
   *   Settings form.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['policy'] = [
      '#config_target' => 'ip_anon.settings:policy',
      '#type' => 'radios',
      '#title' => $this->t('Retention policy'),
      '#options' => [
        $this->t('Preserve IP addresses'),
        $this->t('Anonymize IP addresses'),
      ],
      '#description' => $this->t('This setting may be used to temporarily disable IP anonymization.'),
    ];
    $form['period'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Retention period'),
      '#description' => $this->t('IP addresses older than the retention period will be anonymized.'),
    ];
    $intervals = [
      0,
      30,
      60,
      120,
      180,
      300,
      600,
      900,
      1800,
      2700,
      3600,
      5400,
      7200,
      10800,
      21600,
      32400,
      43200,
      64800,
      86400,
      172800,
      259200,
      345600,
      518400,
      604800,
      1209600,
      2419200,
      4838400,
      7776000,
      9676800,
      31536000,
    ];
    $options = array_combine($intervals, array_map([
      $this->dateFormatter,
      'formatInterval',
    ], $intervals));
    $options[-1] = $this->t('Forever');
    foreach ($this->ipAnonymize->getTables() as $table => $columns) {
      $form['period']["period_$table"] = [
        '#config_target' => "ip_anon.settings:period_$table",
        '#type' => 'select',
        '#title' => $this->t('@table table', ['@table' => $table]),
        '#options' => $options,
        '#description' => new FormattableMarkup('@description', ['@description' => $this->ipAnonymize->getTableDescription($table)]),
      ];
    }
    return parent::buildForm($form, $form_state);
  }

}
