<?php

namespace Drupal\ip_anon;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Session\SessionHandler;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides ip_anon functionality.
 */
class IpAnonymize implements IpAnonymizeInterface {

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected Connection $connection,
    protected ModuleHandlerInterface $moduleHandler,
    protected RequestStack $requestStack,
    protected TimeInterface $time,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function scrub(): void {
    $config = $this->configFactory->get('ip_anon.settings');
    foreach ($this->getTables() as $table => $columns) {
      $period = $config->get("period_$table");
      if (!is_numeric($period) || $period < 0) {
        continue;
      }
      $query = $this->connection->update($table)
        ->fields([$columns['hostname'] => '0'])
        ->condition($columns['timestamp'], \strval($this->time->getRequestTime() - $period), '<=')
        ->condition($columns['hostname'], '0', '<>');
      $rows = $query->execute();
      if (!empty($columns['callback']) && $rows) {
        $columns['callback']();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultColumns(): array {
    $columns = ['hostname', 'timestamp'];
    return array_combine($columns, $columns);
  }

  /**
   * {@inheritdoc}
   */
  public function getTables(): array {
    $tables = [];
    // Sessions table might not exist if an alternative session handler is in
    // use.
    if ($this->connection->schema()->tableExists('sessions')) {
      $tables['sessions'] = $this->getDefaultColumns();
    }
    $this->moduleHandler->alter('ip_anon', $tables);
    // @phpstan-ignore return.type
    return $tables;
  }

  /**
   * {@inheritdoc}
   */
  public function getTableDescription(string $table): string {
    if ($table === 'sessions') {
      $sessionStorage = new class($this->requestStack, $this->connection, $this->time) extends SessionHandler {

        /**
         * Retrieves table description from session schema definition.
         */
        public function getDescription(): string {
          // @phpstan-ignore return.type
          return $this->schemaDefinition()['description'];
        }

      };
      return $sessionStorage->getDescription();
    }
    if (method_exists($this->connection->schema(), 'getComment')) {
      // @phpstan-ignore return.type
      return $this->connection->schema()->getComment($table);
    }
    // Sqlite doesn't have a getComment() method.
    return '';
  }

}
