<?php

namespace Drupal\ip_info\Controller;

use Drupal\ban\BanIpManagerInterface;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Database\Query\TableSortExtender;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\dblog\Controller\DbLogController;
use Drupal\ip_info\Services\CrowdSecInterface;
use Drupal\ip_info\Services\IpApiCoInterface;
use Drupal\ip_info\Services\IpApiIsInterface;
use Drupal\ip_info\Services\IpGeolocationIoInterface;
use Drupal\ip_info\Services\IpInfoInterface;
use Drupal\ip_info\Services\IpInfoIoInterface;
use Drupal\ip_info\Services\IpStackComInterface;
use Drupal\ip_info\Services\WebformSubmissionsListInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for IP address.
 */
class IpController extends DbLogController {

  /**
   * The ban IP manager.
   *
   * @var \Drupal\ban\BanIpManagerInterface
   */
  protected $ipManager;

  /**
   * The CrowdSec service.
   *
   * @var \Drupal\ip_info\Services\CrowdSecInterface
   */
  protected $crowdsec;

  /**
   * The IP Info base service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface
   */
  protected IpInfoInterface $ipInfoBase;

  /**
   * The IPinfo.io service.
   *
   * @var \Drupal\ip_info\Services\IpInfoIoInterface
   */
  protected $ipInfoIo;

  /**
   * The IPapi.is service.
   *
   * @var \Drupal\ip_info\Services\IpApiIsInterface
   */
  protected $ipApiIs;

  /**
   * The IPgeolocation.io service.
   *
   * @var \Drupal\ip_info\Services\IpGeolocationIoInterface
   */
  protected $ipGeolocationIo;

  /**
   * The IPapi.co service.
   *
   * @var \Drupal\ip_info\Services\IpApiCoInterface
   */
  protected $ipApiCo;

  /**
   * The IPstack.com service.
   *
   * @var \Drupal\ip_info\Services\IpStackComInterface
   */
  protected $ipStackCom;

  /**
   * The webform submissions list service.
   *
   * @var \Drupal\ip_info\Services\WebformSubmissionsListInterface
   */
  protected WebformSubmissionsListInterface $webformSubmissionsList;

  /**
   * Constructs a new IpController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\ban\BanIpManagerInterface $ip_manager
   *   The ban IP manager.
   * @param \Drupal\ip_info\Services\IpInfoInterface $ip_info_base
   *   The IP Info base service.
   * @param \Drupal\ip_info\Services\CrowdSecInterface $crowdsec
   *   The CrowdSec service.
   * @param \Drupal\ip_info\Services\IpInfoIoInterface $ipinfo_io
   *   The IPinfo.io service.
   * @param \Drupal\ip_info\Services\IpApiIsInterface $ipapi_is
   *   The IPapi.is service.
   * @param \Drupal\ip_info\Services\IpGeolocationIoInterface $ip_geolocation_io
   *   The IPgeolocation.io service.
   * @param \Drupal\ip_info\Services\IpApiCoInterface $ip_api_co
   *   The IPapi.co service.
   * @param \Drupal\ip_info\Services\IpStackComInterface $ip_stack_com
   *   The IPstack.com service.
   * @param \Drupal\ip_info\Services\WebformSubmissionsListInterface $webform_submissions_list
   *   The webform submissions list service.
   */
  public function __construct(
    Connection $database,
    ModuleHandlerInterface $module_handler,
    DateFormatterInterface $date_formatter,
    FormBuilderInterface $form_builder,
    BanIpManagerInterface $ip_manager,
    IpInfoInterface $ip_info_base,
    CrowdSecInterface $crowdsec,
    IpInfoIoInterface $ipinfo_io,
    IpApiIsInterface $ipapi_is,
    IpGeolocationIoInterface $ip_geolocation_io,
    IpApiCoInterface $ip_api_co,
    IpStackComInterface $ip_stack_com,
    WebformSubmissionsListInterface $webform_submissions_list,
  ) {
    parent::__construct($database, $module_handler, $date_formatter, $form_builder);
    $this->ipManager = $ip_manager;
    $this->ipInfoBase = $ip_info_base;
    $this->crowdsec = $crowdsec;
    $this->ipInfoIo = $ipinfo_io;
    $this->ipApiIs = $ipapi_is;
    $this->ipGeolocationIo = $ip_geolocation_io;
    $this->ipApiCo = $ip_api_co;
    $this->ipStackCom = $ip_stack_com;
    $this->webformSubmissionsList = $webform_submissions_list;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('module_handler'),
      $container->get('date.formatter'),
      $container->get('form_builder'),
      $container->get('ban.ip_manager'),
      $container->get('ip_info.base'),
      $container->get('ip_info.crowdsec'),
      $container->get('ip_info.ipinfo_io'),
      $container->get('ip_info.ipapi_is'),
      $container->get('ip_info.ipgeolocation_io'),
      $container->get('ip_info.ipapi_co'),
      $container->get('ip_info.ipstack_com'),
      $container->get('ip_info.webform_submissions_list'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function overview($ip_address) {

    // Test, if the IP address is banned.
    if ($this->ipManager->isBanned($ip_address)) {
      $build['dblog_filter_form']['#prefix'] = '<p>' . $this->t('The IP address is banned.') . ' ';
      // Find the ID of the banned IP.
      $bans = $this->ipManager->findAll()->fetchAll();
      foreach ($bans as $ban) {
        if ($ban->ip == $ip_address) {
          $ban_id = $ban->iid;
          break;
        }
      }
      // Add unban link.
      if (isset($ban_id)) {
        $build['dblog_filter_form']['#prefix'] .= Link::fromTextAndUrl($this->t('Unban this IP address'), Url::fromRoute('ban.delete', ['ban_id' => $ban_id]))->toString() . '.</p>';
      }
      else {
        $build['dblog_filter_form']['#prefix'] .= '</p>';
      }
    }
    else {
      $build['dblog_filter_form']['#prefix'] = '<p>' . $this->t('The IP address is not banned.') . ' ';
      // Add ban link.
      $build['dblog_filter_form']['#prefix'] .= Link::fromTextAndUrl($this->t('Ban this IP address'), Url::fromRoute('ban.admin_page', ['default_ip' => $ip_address]))->toString() . '.</p>';
    }

    // DNS Lookup.
    $lookup_services = [
      'db-ip.com' => 'https://db-ip.com/' . $ip_address,
      'ipapi.co' => 'https://ipapi.co/?q=' . $ip_address,
      'ipinfo.io' => 'https://ipinfo.io/' . $ip_address,
      'iplocation.net' => 'https://www.iplocation.net/ip-lookup/' . $ip_address,
      'ipregistry.co' => 'https://ipregistry.co/' . $ip_address,
      'ip2location.com' => 'https://www.ip2location.com/demo/' . $ip_address,
      'keycdn.com' => 'https://tools.keycdn.com/geo?host=' . $ip_address,
    ];
    $lookup_services_links = '';
    foreach ($lookup_services as $service => $url) {
      $lookup_services_links .= '<a href="' . $url . '">' . $service . '</a>, ';
    }
    // Remove the last comma.
    $lookup_services_links = rtrim($lookup_services_links, ', ');
    $build['dblog_filter_form']['#prefix'] .= '<p>' . $this->t('Lookup the IP:') . ' ' . $lookup_services_links . '</p>';

    // Get IP data from the CrowdSec CTI API.
    if (!empty($crowdsec_data = $this->crowdsec->getAllData($ip_address))) {
      $build['cti_label'] = [
        '#type' => 'markup',
        '#markup' => '<h3>' . $this->t('CrowdSec CTI data') . '</h3>',
      ];
      $build['cti_table'] = [
        '#type' => 'table',
        '#rows' => $crowdsec_data,
        '#attributes' => ['id' => 'admin-cti', 'class' => ['admin-list']],
      ];
    }

    // Get IP data from IPinfo.io API.
    if (!empty($ipinfo_data = $this->ipInfoIo->getProcessedData($ip_address))) {
      $build['ipinfo_label'] = [
        '#type' => 'markup',
        '#markup' => '<h3>' . $this->t('IPinfo.io data') . '</h3>',
      ];
      $build['ipinfo_table'] = [
        '#type' => 'table',
        '#rows' => $ipinfo_data,
        '#attributes' => ['id' => 'admin-ipinfo', 'class' => ['admin-list']],
      ];
    }

    // Get IP data from ipapi.is API.
    if (!empty($ipapi_data = $this->ipApiIs->getProcessedData($ip_address))) {
      $build['ipapi_label'] = [
        '#type' => 'markup',
        '#markup' => '<h3>' . $this->t('ipapi.is data') . '</h3>',
      ];
      $build['ipapi_table'] = [
        '#type' => 'table',
        '#rows' => $ipapi_data,
        '#attributes' => ['id' => 'admin-ipapi', 'class' => ['admin-list']],
      ];
    }

    // Get IP data from ipgeolocation.io API.
    if (!empty($ipgeolocation_data = $this->ipGeolocationIo->getProcessedData($ip_address))) {
      $build['ipgeolocation_label'] = [
        '#type' => 'markup',
        '#markup' => '<h3>' . $this->t('ipgeolocation.io data') . '</h3>',
      ];
      $build['ipgeolocation_table'] = [
        '#type' => 'table',
        '#rows' => $ipgeolocation_data,
        '#attributes' => ['id' => 'admin-ipgeolocation', 'class' => ['admin-list']],
      ];
    }

    // Get IP data from ipapi.co API.
    if (!empty($ipapi_co_data = $this->ipApiCo->getProcessedData($ip_address))) {
      $build['ipapi_co_label'] = [
        '#type' => 'markup',
        '#markup' => '<h3>' . $this->t('ipapi.co data') . '</h3>',
      ];
      $build['ipapi_co_table'] = [
        '#type' => 'table',
        '#rows' => $ipapi_co_data,
        '#attributes' => ['id' => 'admin-ipapi-co', 'class' => ['admin-list']],
      ];
    }

    // Get IP data from ipstack.com API.
    if (!empty($ipstack_com_data = $this->ipStackCom->getProcessedData($ip_address))) {
      $build['ipstack_com_label'] = [
        '#type' => 'markup',
        '#markup' => '<h3>' . $this->t('ipstack.com data') . '</h3>',
      ];
      $build['ipstack_com_table'] = [
        '#type' => 'table',
        '#rows' => $ipstack_com_data,
        '#attributes' => ['id' => 'admin-ipstack-com', 'class' => ['admin-list']],
      ];
    }

    $build['dblog_label'] = [
      '#type' => 'markup',
      '#markup' => '<h3>' . $this->t('Log messages') . '</h3>',
    ];

    // Add log message statistics.
    $log_stats = $this->buildLogMessageStats($ip_address);
    if (!empty($log_stats)) {
      $build['dblog_stats'] = $log_stats;
    }

    $dblog_rows = [];

    $classes = static::getLogLevelClassMap();

    $dblog_header = [
      // Icon column.
      '',
      [
        'data' => $this->t('Type'),
        'field' => 'w.type',
        'class' => [RESPONSIVE_PRIORITY_MEDIUM],
      ],
      [
        'data' => $this->t('Date'),
        'field' => 'w.wid',
        'sort' => 'desc',
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      $this->t('Message'),
      [
        'data' => $this->t('User'),
        'field' => 'ufd.name',
        'class' => [RESPONSIVE_PRIORITY_MEDIUM],
      ],
    ];

    $query = $this->database->select('watchdog', 'w')
      ->extend(PagerSelectExtender::class)
      ->extend(TableSortExtender::class);
    $query->fields('w', [
      'wid',
      'uid',
      'severity',
      'type',
      'hostname',
      'timestamp',
      'message',
      'variables',
      'link',
    ]);
    $query->leftJoin('users_field_data', 'ufd', '[w].[uid] = [ufd].[uid]');

    $result = $query
      ->limit(50)
      ->condition('hostname', $ip_address)
      ->orderByHeader($dblog_header)
      ->execute();

    foreach ($result as $dblog) {
      $message = $this->formatMessage($dblog);
      if ($message && isset($dblog->wid)) {
        $title = Unicode::truncate(Html::decodeEntities(strip_tags($message)), 256, TRUE, TRUE);
        $log_text = Unicode::truncate($title, 56, TRUE, TRUE);
        // The link generator will escape any unsafe HTML entities in the final
        // text.
        $message = Link::fromTextAndUrl($log_text, new Url('dblog.event', ['event_id' => $dblog->wid], [
          'attributes' => [
            // Provide a title for the link for useful hover hints. The
            // Attribute object will escape any unsafe HTML entities in the
            // final text.
            'title' => $title,
          ],
        ]))->toString();
      }
      $username = [
        '#theme' => 'username',
        '#account' => $this->userStorage->load($dblog->uid),
      ];
      $dblog_rows[] = [
        'data' => [
          // Cells.
          ['class' => ['icon']],
          $dblog->type,
          $this->dateFormatter->format($dblog->timestamp, 'short'),
          $message,
          ['data' => $username],
          ['data' => ['#markup' => $dblog->link]],
        ],
        // Attributes for table row.
        'class' => [Html::getClass('dblog-' . $dblog->type), $classes[$dblog->severity]],
      ];
    }

    $build['dblog_table'] = [
      '#type' => 'table',
      '#header' => $dblog_header,
      '#rows' => $dblog_rows,
      '#attributes' => ['id' => 'admin-dblog', 'class' => ['admin-dblog']],
      '#empty' => $this->t('No log messages available.'),
      '#attached' => [
        'library' => ['dblog/drupal.dblog'],
      ],
    ];
    $build['dblog_pager'] = ['#type' => 'pager'];

    // Add webform submissions if the webform module is available.
    if ($this->webformSubmissionsList->isWebformModuleAvailable()) {
      $webform_submissions_build = $this->webformSubmissionsList->buildSubmissionsList($ip_address);
      if (!empty($webform_submissions_build)) {
        $build['webform_submissions_label'] = [
          '#type' => 'markup',
          '#markup' => '<h3>' . $this->t('Webform Submissions') . '</h3>',
        ];

        // Add submission statistics.
        $stats = $this->webformSubmissionsList->buildSubmissionStats($ip_address);
        if (!empty($stats)) {
          $build['webform_submissions_stats'] = $stats;
        }

        $build += $webform_submissions_build;
      }
    }

    return $build;

  }

  /**
   * {@inheritdoc}
   */
  public function eventDetails($event_id) {
    $build = parent::eventDetails($event_id);
    $ip_address = $build["dblog_table"]["#rows"][7][1];

    if ($this->ipInfoBase->isIpv4($ip_address)) {
      $build["dblog_table"]["#rows"][7][1] = Link::fromTextAndUrl(
        $ip_address,
        Url::fromRoute('ip_info.ip', ['ip_address' => $ip_address])
      );
    }

    return $build;
  }

  /**
   * Build log message statistics for an IP address.
   *
   * @param string $ip_address
   *   The IP address to build statistics for.
   *
   * @return array
   *   Render array containing the statistics.
   */
  protected function buildLogMessageStats(string $ip_address): array {
    $total = $this->getTotalLogCount($ip_address);

    if ($total == 0) {
      return [];
    }

    $severity_counts = $this->getLogCountsBySeverity($ip_address);
    $date_range = $this->getLogDateRange($ip_address);
    $unique_types_count = $this->getUniqueLogTypesCount($ip_address);

    $stats_items = [];
    $stats_items[] = $this->t('<strong>Total messages:</strong> @total', ['@total' => $total]);

    // Add severity breakdown.
    $severity_labels = static::getLogLevelClassMap();
    $severity_names = [
      0 => $this->t('Emergency'),
      1 => $this->t('Alert'),
      2 => $this->t('Critical'),
      3 => $this->t('Error'),
      4 => $this->t('Warning'),
      5 => $this->t('Notice'),
      6 => $this->t('Info'),
      7 => $this->t('Debug'),
    ];

    $severity_stats = [];
    foreach ($severity_counts as $severity => $count) {
      if ($count > 0) {
        $name = $severity_names[$severity] ?? $this->t('Level @level', ['@level' => $severity]);
        $severity_stats[] = $this->t('@name: @count', ['@name' => $name, '@count' => $count]);
      }
    }
    if (!empty($severity_stats)) {
      $stats_items[] = implode(', ', $severity_stats);
    }

    if ($date_range) {
      $stats_items[] = $this->t('<strong>Date range:</strong> @range', ['@range' => $date_range]);
    }

    $stats_items[] = $this->t('<strong>Unique types:</strong> @count', ['@count' => $unique_types_count]);

    return [
      '#type' => 'markup',
      '#markup' => '<div class="log-message-stats"><p>' . implode(' | ', $stats_items) . '</p></div>',
    ];
  }

  /**
   * Get total log message count for an IP address.
   *
   * @param string $ip_address
   *   The IP address to count messages for.
   *
   * @return int
   *   The total number of log messages.
   */
  protected function getTotalLogCount(string $ip_address): int {
    $query = $this->database->select('watchdog', 'w');
    $query->condition('hostname', $ip_address);
    $query = $query->countQuery();

    return (int) $query->execute()->fetchField();
  }

  /**
   * Get log message counts by severity level for an IP address.
   *
   * @param string $ip_address
   *   The IP address to count messages for.
   *
   * @return array
   *   Array keyed by severity level with count values.
   */
  protected function getLogCountsBySeverity(string $ip_address): array {
    $query = $this->database->select('watchdog', 'w');
    $query->condition('hostname', $ip_address);
    $query->fields('w', ['severity']);
    $query->addExpression('COUNT(*)', 'count');
    $query->groupBy('severity');
    $query->orderBy('severity');

    $result = $query->execute();
    $counts = [];

    // Initialize all severity levels to 0.
    for ($i = 0; $i <= 7; $i++) {
      $counts[$i] = 0;
    }

    // Fill in actual counts.
    while ($row = $result->fetchAssoc()) {
      $counts[$row['severity']] = (int) $row['count'];
    }

    return $counts;
  }

  /**
   * Get the date range of log messages for an IP address.
   *
   * @param string $ip_address
   *   The IP address to get date range for.
   *
   * @return string
   *   Formatted date range string.
   */
  protected function getLogDateRange(string $ip_address): string {
    // Get oldest message.
    $oldest_query = $this->database->select('watchdog', 'w');
    $oldest_query->condition('hostname', $ip_address);
    $oldest_query->fields('w', ['timestamp']);
    $oldest_query->orderBy('timestamp', 'ASC');
    $oldest_query->range(0, 1);

    $oldest_result = $oldest_query->execute()->fetchField();

    // Get newest message.
    $newest_query = $this->database->select('watchdog', 'w');
    $newest_query->condition('hostname', $ip_address);
    $newest_query->fields('w', ['timestamp']);
    $newest_query->orderBy('timestamp', 'DESC');
    $newest_query->range(0, 1);

    $newest_result = $newest_query->execute()->fetchField();

    if (!$oldest_result || !$newest_result) {
      return '';
    }

    $oldest_date = $this->dateFormatter->format($oldest_result, 'short');
    $newest_date = $this->dateFormatter->format($newest_result, 'short');

    if ($oldest_result == $newest_result) {
      return $newest_date;
    }

    return $oldest_date . ' - ' . $newest_date;
  }

  /**
   * Get the number of unique log message types for an IP address.
   *
   * @param string $ip_address
   *   The IP address to count unique types for.
   *
   * @return int
   *   The number of unique message types.
   */
  protected function getUniqueLogTypesCount(string $ip_address): int {
    $query = $this->database->select('watchdog', 'w');
    $query->condition('hostname', $ip_address);
    $query->addExpression('COUNT(DISTINCT type)', 'unique_count');

    $result = $query->execute()->fetchField();
    return (int) $result;
  }

  /**
   * Generates the page title.
   *
   * @param string $ip_address
   *   The IP address from the path.
   *
   * @return string
   *   The page title.
   */
  public function title($ip_address) {
    return 'IP Address Info for: ' . $ip_address;
  }

}
