<?php

declare(strict_types=1);

namespace Drupal\ip_info\Form;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\ip_info\Services\IpInfoInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure IP Info settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a new SettingsForm object.
   *
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $urlGenerator
   *   The URL generator service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\ip_info\Services\IpInfoInterface $ipInfoBase
   *   The IP Info service.
   */
  public function __construct(
    protected readonly UrlGeneratorInterface $urlGenerator,
    protected readonly ModuleHandlerInterface $moduleHandler,
    protected readonly IpInfoInterface $ipInfoBase,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('url_generator'),
      $container->get('module_handler'),
      $container->get('ip_info.base'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ip_info_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ip_info.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['ipinfo_io_token'] = [
      '#type' => 'key_select',
      '#title' => $this->t('IPinfo.io token'),
      '#description' => $this->t('(You can get a token from <a href="https://ipinfo.io" target="_blank">ipinfo.io</a>).'),
      '#default_value' => $this->config('ip_info.settings')->get('ipinfo_io_token'),
    ];
    $form['ipapi_is_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('ipapi.is key'),
      '#description' => $this->t('(You can get a key from <a href="https://ipapi.is" target="_blank">ipapi.is</a>).'),
      '#default_value' => $this->config('ip_info.settings')->get('ipapi_is_key'),
    ];
    $form['ipgeolocation_io_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('ipgeolocation.io key'),
      '#description' => $this->t('(You can get a key from <a href="https://ipgeolocation.io" target="_blank">ipgeolocation.io</a>).'),
      '#default_value' => $this->config('ip_info.settings')->get('ipgeolocation_io_key'),
    ];
    $form['ipstack_com_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('ipstack.com key'),
      '#description' => $this->t('(You can get a key from <a href="https://ipstack.com" target="_blank">ipstack.com</a>).'),
      '#default_value' => $this->config('ip_info.settings')->get('ipstack_com_key'),
    ];
    if ($this->ipInfoBase->settingExists('ipapi_co_enable', 'bool')) {
      $form['ipapi_co_enable'] = [
        '#type' => 'item',
        '#markup' => $this->t('The ipapi.co is already set in the settings.'),
      ];
    }
    else {
      $form['ipapi_co_enable'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable ipapi.co'),
        '#default_value' => $this->config('ip_info.settings')->get('ipapi_co_enable'),
        '#description' => $this->t('Check this box if you want to enable <a href="https://ipapi.co" target="_blank">ipapi.co</a>.'),
      ];
    }
    // Add link to the CrowdSec module configuration, if the module is enabled.
    if ($this->moduleHandler->moduleExists('crowdsec')) {
      $form['crowdsec'] = [
        '#type' => 'item',
        '#markup' => $this->t('If you want to display IP data from CrowdSec database, add your \'CTI API key\' to the <a href=":url">CrowdSec module configuration</a>.', [
          ':url' => $this->urlGenerator->generateFromRoute('crowdsec.settings'),
        ]),
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('ip_info.settings')
      ->set('ipinfo_io_token', $form_state->getValue('ipinfo_io_token'))
      ->set('ipapi_is_key', $form_state->getValue('ipapi_is_key'))
      ->set('ipgeolocation_io_key', $form_state->getValue('ipgeolocation_io_key'))
      ->set('ipstack_com_key', $form_state->getValue('ipstack_com_key'))
      ->set('ipapi_co_enable', $form_state->getValue('ipapi_co_enable'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
