<?php

declare(strict_types=1);

namespace Drupal\ip_info\Hook;

use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Hook implementations for webform-related functionality.
 */
class IpInfoWebformHooks {

  /**
   * Implements hook_preprocess_table() for webform submissions.
   */
  public function preprocessTable(array &$variables): void {
    // Check if this is a webform submissions table.
    if (!isset($variables['attributes']['class']) ||
        !in_array('webform-results-table', $variables['attributes']['class'])) {
      return;
    }

    // Process each row to enhance IP addresses.
    foreach ($variables['rows'] as $row_index => &$row) {
      $this->processTableRow($row);
    }
  }

  /**
   * Process a single table row to enhance IP address columns.
   *
   * @return bool
   *   TRUE if any IP addresses were processed, FALSE otherwise.
   */
  private function processTableRow(array &$row): bool {
    $processed = FALSE;
    foreach ($row['cells'] as $cell_index => &$cell) {
      if ($this->isIpAddressCell($cell)) {
        $ip_address = $cell['content'];
        $cell['content'] = $this->createIpInfoLink($ip_address);
        $processed = TRUE;
      }
    }
    return $processed;
  }

  /**
   * Check if a table cell contains an IP address.
   */
  private function isIpAddressCell(array $cell): bool {
    return isset($cell['content']) &&
           is_string($cell['content']) &&
           filter_var($cell['content'], FILTER_VALIDATE_IP);
  }

  /**
   * Create a link to the IP info page.
   */
  private function createIpInfoLink(string $ip_address): array {
    $ip_link = Link::fromTextAndUrl(
      $ip_address,
      Url::fromRoute('ip_info.ip', ['ip_address' => $ip_address])
    );

    return $ip_link->toRenderable();
  }

}
