<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crowdsec\CtiInterface;

/**
 * Provides a CrowdSec service.
 */
class CrowdSec implements CrowdSecInterface {

  use StringTranslationTrait;

  /**
   * The CrowdSec CTI service.
   *
   * @var \Drupal\crowdsec\CtiInterface
   */
  protected ?CtiInterface $cti = NULL;

  /**
   * Constructs a CrowdSec object.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter.
   */
  public function __construct(
    protected readonly DateFormatterInterface $dateFormatter,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function setCtiService(?CtiInterface $cti = NULL): void {
    $this->cti = $cti;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllData(string $ip): array {
    $data = [];
    if (isset($this->cti)) {
      $smoke = $this->cti->getSmoke($ip);
    }
    else {
      return $data;
    }
    if (empty($smoke)) {
      return $data;
    }
    if ($basic_info = $this->getBasicInfo($smoke)) {
      $data[] = ['label' => $this->t('Basic info'), 'value' => $basic_info];
    }
    if ($background_noise = $this->getBackgroundNoise($smoke)) {
      $data[] = ['label' => $this->t('Background noise'), 'value' => $background_noise];
    }
    if ($location = $this->getLocation($smoke)) {
      $data[] = ['label' => $this->t('Location'), 'value' => $location];
    }
    if ($organization = $this->getOrganization($smoke)) {
      $data[] = ['label' => $this->t('Organization'), 'value' => $organization];
    }
    if ($ip_range = $this->getIpRange($smoke)) {
      $data[] = ['label' => $this->t('IP range'), 'value' => $ip_range];
    }
    if ($ip_range24 = $this->getIpRange24($smoke)) {
      $data[] = ['label' => $this->t('IP range /24'), 'value' => $ip_range24];
    }
    if ($behaviors = $this->getBehaviors($smoke)) {
      $data[] = ['label' => $this->t('Behaviors'), 'value' => $behaviors];
    }
    if ($attack_details = $this->getAttackDetails($smoke)) {
      $data[] = ['label' => $this->t('Attack details'), 'value' => $attack_details];
    }
    if ($target_countries = $this->getTargetCountries($smoke)) {
      $data[] = ['label' => $this->t('Target countries'), 'value' => $target_countries];
    }
    if ($history = $this->getHistory($smoke)) {
      $data[] = ['label' => $this->t('History'), 'value' => $history];
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  private function getBasicInfo(array $smoke): string {
    $value = '';
    if (array_key_exists('reputation', $smoke)) {
      $value .= 'Reputation: ' . $smoke['reputation'];
    }
    if (array_key_exists('ip_range_reputation', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= 'IP range reputation: ' . $smoke['ip_range_reputation'];
    }
    if (array_key_exists('confidence', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= 'Confidence: ' . $smoke['confidence'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getBackgroundNoise(array $smoke): string {
    $value = '';
    if (array_key_exists('background_noise', $smoke)) {
      $value .= $smoke['background_noise'];
    }
    if (array_key_exists('background_noise_score', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= 'score: ' . $smoke['background_noise_score'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getLocation(array $smoke): string {
    $value = '';
    if (array_key_exists('location', $smoke)) {
      $value .= $smoke['location']['city'] . ', ' . $smoke['location']['country'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getOrganization(array $smoke): string {
    $value = '';
    if (array_key_exists('as_num', $smoke)) {
      $value .= 'AS' . $smoke['as_num'];
    }
    if (array_key_exists('as_name', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= $smoke['as_name'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getIpRange(array $smoke): string {
    $value = '';
    if (array_key_exists('ip_range', $smoke)) {
      $value .= $smoke['ip_range'];
    }
    if (array_key_exists('ip_range_reputation', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= 'reputation: ' . $smoke['ip_range_reputation'];
    }
    if (array_key_exists('ip_range_score', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= 'score: ' . $smoke['ip_range_score'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getIpRange24(array $smoke): string {
    $value = '';
    if (array_key_exists('ip_range24', $smoke)) {
      $value .= $smoke['ip_range24'];
    }
    if (array_key_exists('ip_range24_reputation', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= 'reputation: ' . $smoke['ip_range24_reputation'];
    }
    if (array_key_exists('ip_range24_score', $smoke)) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= 'score: ' . $smoke['ip_range24_score'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getBehaviors(array $smoke): string {
    $value = '';
    foreach ($smoke['behaviors'] as $behavior) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= $behavior['label'] . ' (' . $behavior['description'] . ')';
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getAttackDetails(array $smoke): string {
    $value = '';
    foreach ($smoke['attack_details'] as $attack_detail) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= $attack_detail['label'] . ' (' . $attack_detail['description'] . ')';
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getTargetCountries(array $smoke): string {
    $value = '';
    foreach ($smoke['target_countries'] as $target_country => $count) {
      if (!empty($value)) {
        $value .= '; ';
      }
      $value .= $target_country . ' (' . $count . ')';
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  private function getHistory(array $smoke): TranslatableMarkup {
    $first_seen = $this->dateFormatter->format(strtotime($smoke['history']['first_seen']), 'short');
    $last_seen = $this->dateFormatter->format(strtotime($smoke['history']['last_seen']), 'short');

    return $this->t('First seen: @first_seen; Last seen: @last_seen', [
      '@first_seen' => $first_seen,
      '@last_seen' => $last_seen,
    ]);
  }

}
