<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides a service for interacting with the ipapi.co API.
 */
class IpApiCo implements IpApiCoInterface {

  use StringTranslationTrait;

  /**
   * The base URL for the ipapi.co API.
   */
  protected const API_BASE_URL = 'https://ipapi.co/';

  /**
   * The IP Info base service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface
   */
  protected IpInfoInterface $ipInfoBase;

  /**
   * Constructs a new IpApiCo service.
   *
   * @param \Drupal\ip_info\Services\IpInfoInterface $ip_info_base
   *   The IP Info base service.
   */
  public function __construct(IpInfoInterface $ip_info_base) {
    $this->ipInfoBase = $ip_info_base;
  }

  /**
   * {@inheritdoc}
   */
  public function getProcessedData(string $ip_address): array {
    $data = [];

    if (!$this->ipInfoBase->isEnabled('ipapi_co_enable')) {
      return $data;
    }

    $url = self::API_BASE_URL . $ip_address . '/json';

    $ip_information = $this->ipInfoBase->getIpInformation($ip_address, $url, 'ipapi.co');

    if ($ip_information) {
      if (array_key_exists('city', $ip_information)) {
        $location_info = $ip_information['city'];
        if (array_key_exists('region', $ip_information)) {
          $location_info .= ', ' . $ip_information['region'];
        }
        if (array_key_exists('country_name', $ip_information)) {
          $location_info .= ', ' . $ip_information['country_name'];
        }
        if (array_key_exists('in_eu', $ip_information) && $ip_information['in_eu'] === TRUE) {
          $location_info .= ' (EU)';
        }
        $data[] = ['label' => $this->t('Location'), 'value' => $location_info];
      }
      if (array_key_exists('org', $ip_information)) {
        $organization_info = $ip_information['org'];
        if (array_key_exists('asn', $ip_information)) {
          $organization_info = $ip_information['asn'] . ' ' . $organization_info;
        }
        $data[] = ['label' => $this->t('Organization'), 'value' => $organization_info];
      }
    }

    return $data;
  }

}
