<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides a service for interacting with the ipapi.is API.
 */
class IpApiIs implements IpApiIsInterface {

  use StringTranslationTrait;


  /**
   * The base URL for the ipapi.is API.
   */
  protected const API_BASE_URL = 'https://api.ipapi.is/';

  /**
   * The IP Info base service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface
   */
  protected IpInfoInterface $ipInfoBase;

  /**
   * Constructs a new IpApiIs service.
   *
   * @param \Drupal\ip_info\Services\IpInfoInterface $ip_info_base
   *   The IP Info base service.
   */
  public function __construct(IpInfoInterface $ip_info_base) {
    $this->ipInfoBase = $ip_info_base;
  }

  /**
   * {@inheritdoc}
   */
  public function getProcessedData(string $ip_address): array {
    $data = [];
    $key = $this->ipInfoBase->getApiKey('ipapi_is_key');

    if (empty($key)) {
      return $data;
    }

    $url = self::API_BASE_URL . '?q=' . $ip_address . '&key=' . $key;

    $ip_information = $this->ipInfoBase->getIpInformation($ip_address, $url, 'ipapi.is');

    if ($ip_information) {
      if (array_key_exists('location', $ip_information)) {
        $location_info = "";
        if (array_key_exists('city', $ip_information['location'])) {
          $location_info .= $ip_information['location']['city'];
        }
        if (array_key_exists('zip', $ip_information['location'])) {
          $location_info .= " (" . $ip_information['location']['zip'] . ")";
        }
        if (array_key_exists('state', $ip_information['location'])) {
          $location_info .= ", " . $ip_information['location']['state'];
        }
        if (array_key_exists('country', $ip_information['location'])) {
          $location_info .= ", " . $ip_information['location']['country'];
        }
        $data[] = ['label' => $this->t('Location'), 'value' => $location_info];
      }

      if (array_key_exists('company', $ip_information)) {
        $company_info = "";
        if (array_key_exists('name', $ip_information['company'])) {
          $company_info .= $ip_information['company']['name'];
        }
        if (array_key_exists('type', $ip_information['company'])) {
          $company_info .= " - " . $ip_information['company']['type'];
        }
        if (array_key_exists('domain', $ip_information['company'])) {
          $company_info .= " (" . $ip_information['company']['domain'] . ")";
        }
        if (array_key_exists('abuser_score', $ip_information['company'])) {
          $company_info .= '; abuser score: ' . $ip_information['company']['abuser_score'];
        }
        $data[] = ['label' => $this->t('Company'), 'value' => $company_info];
      }

      if (array_key_exists('datacenter', $ip_information)) {
        $data_center_info = "";
        if (array_key_exists('datacenter', $ip_information['datacenter'])) {
          $data_center_info .= $ip_information['datacenter']['datacenter'];
        }
        if (array_key_exists('domain', $ip_information['datacenter'])) {
          $data_center_info .= " (" . $ip_information['datacenter']['domain'] . ")";
        }
        $data[] = ['label' => $this->t('Data center'), 'value' => $data_center_info];
      }

      $ip_properties = [
        "is_bogon" => "Bogon",
        "is_mobile" => "Mobile",
        "is_satellite" => "Satellite",
        "is_crawler" => "Crawler",
        "is_datacenter" => "Datacenter",
        "is_tor" => "Tor",
        "is_proxy" => "Proxy",
        "is_vpn" => "VPN",
        "is_abuser" => "Abuser",
      ];
      $ip_is = "";
      foreach ($ip_properties as $ip_property => $ip_property_label) {
        if (array_key_exists($ip_property, $ip_information)) {
          if (!empty($ip_is)) {
            $ip_is .= ", ";
          }
          $ip_is .= $ip_property_label;
        }
      }
      if (!empty($ip_is)) {
        $data[] = ['label' => $this->t('The IP is'), 'value' => $ip_is];
      }
    }

    return $data;
  }

}
