<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides a service for interacting with the ipgeolocation.io API.
 */
class IpGeolocationIo implements IpGeolocationIoInterface {

  use StringTranslationTrait;


  /**
   * The base URL for the ipgeolocation.io API.
   */
  protected const API_BASE_URL = 'https://api.ipgeolocation.io/';

  /**
   * The IP Info base service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface
   */
  protected IpInfoInterface $ipInfoBase;

  /**
   * Constructs a new IpGeolocationIo service.
   *
   * @param \Drupal\ip_info\Services\IpInfoInterface $ip_info_base
   *   The IP Info base service.
   */
  public function __construct(IpInfoInterface $ip_info_base) {
    $this->ipInfoBase = $ip_info_base;
  }

  /**
   * {@inheritdoc}
   */
  public function getProcessedData(string $ip_address): array {
    $data = [];
    $key = $this->ipInfoBase->getApiKey('ipgeolocation_io_key');

    if (empty($key)) {
      return $data;
    }

    $url = self::API_BASE_URL . 'ipgeo?apiKey=' . $key . '&ip=' . $ip_address;

    $ip_information = $this->ipInfoBase->getIpInformation($ip_address, $url, 'ipgeolocation.io');

    if ($ip_information) {
      if (array_key_exists('city', $ip_information)) {
        $location_info = $ip_information['city'];
        if (array_key_exists('state_prov', $ip_information)) {
          $location_info .= ', ' . $ip_information['state_prov'];
        }
        if (array_key_exists('country_name', $ip_information)) {
          $location_info .= ', ' . $ip_information['country_name'];
        }
        if (array_key_exists('country_emoji', $ip_information)) {
          $location_info .= ' ' . $ip_information['country_emoji'];
        }
        if (array_key_exists('is_eu', $ip_information) && $ip_information['is_eu'] === TRUE) {
          $location_info .= ' (EU)';
        }
        $data[] = ['label' => $this->t('Location'), 'value' => $location_info];
      }
      if (array_key_exists('organization', $ip_information)) {
        $organization_info = $ip_information['organization'];
        if (array_key_exists('asn', $ip_information)) {
          $organization_info = $ip_information['asn'] . ' ' . $organization_info;
        }
        $data[] = ['label' => $this->t('Organization'), 'value' => $organization_info];
      }
      if (array_key_exists('isp', $ip_information)) {
        $connection_info = $ip_information['isp'];
        if (array_key_exists('connection_type', $ip_information) && !empty($ip_information['connection_type'])) {
          $connection_info .= ', connection type: ' . $ip_information['connection_type'];
        }
        if (array_key_exists('hostname', $ip_information)) {
          $connection_info .= ', hostname: ' . $ip_information['hostname'];
        }
        $data[] = ['label' => $this->t('ISP'), 'value' => $connection_info];
      }
    }

    return $data;
  }

}
