<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Site\Settings;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides a service for interacting with the IP.
 */
class IpInfo implements IpInfoInterface {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected Client $httpClient;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new IpInfo service.
   *
   * @param \GuzzleHttp\Client $http_client
   *   The HTTP client.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    Client $http_client,
    KeyRepositoryInterface $key_repository,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->httpClient = $http_client;
    $this->keyRepository = $key_repository;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('ip_info');
  }

  /**
   * {@inheritdoc}
   */
  public function getIpInformation(string $ip_address, string $url, string $api_name): ?array {
    try {
      $response = $this->httpClient->request('GET', $url);
      $status_code = $response->getStatusCode();

      if ($status_code === Response::HTTP_OK) {
        $data = json_decode($response->getBody()->getContents(), TRUE);
        return $data ?: NULL;
      }
      else {
        $this->logger->error('@api_name API request failed with status code @code for ip address @ip', [
          '@api_name' => $api_name,
          '@code' => $status_code,
          '@ip' => $ip_address,
        ]);
        return NULL;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('@api_name API request failed with error: @error', [
        '@api_name' => $api_name,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getApiKey(string $config_name): ?string {
    $config = $this->configFactory->get('ip_info.settings');
    if ($key_id = $config->get($config_name)) {
      return $this->keyRepository->getKey($key_id)?->getKeyValue();
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function settingExists(string $config_name, $data_type = 'string'): bool {
    $settings = Settings::get('ip_info');
    if ($settings && array_key_exists($config_name, $settings)) {
      // Test if the setting is of the correct data type.
      switch ($data_type) {
        case 'int':
          return is_int($settings[$config_name]);

        case 'bool':
          return is_bool($settings[$config_name]);

        case 'string':
          return is_string($settings[$config_name]);

        case 'array':
          return is_array($settings[$config_name]);

        case 'null':
          return is_null($settings[$config_name]);
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(string $config_name): bool {
    $settings = Settings::get('ip_info');
    $config = $this->configFactory->get('ip_info.settings');
    if ($this->settingExists($config_name) && boolval($settings[$config_name])) {
      return TRUE;
    }
    else {
      return boolval($config->get($config_name));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isIpv4(mixed $ip_address): bool {
    return filter_var($ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) !== FALSE;
  }

}
