<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

/**
 * Interface for the IP Info services.
 */
interface IpInfoInterface {

  /**
   * Gets the IP information from the API.
   *
   * @param string $ip
   *   The IP address to get information for.
   * @param string $url
   *   The URL to get information from.
   * @param string $api_name
   *   The name of the API.
   *
   * @return array|null
   *   An associative array of IP information, or NULL if an error occurred.
   */
  public function getIpInformation(string $ip, string $url, string $api_name): ?array;

  /**
   * Gets the API key from the configuration.
   *
   * @param string $config_name
   *   The name of the configuration/setting.
   *
   * @return string|null
   *   The API key, or NULL if it is not set.
   */
  public function getApiKey(string $config_name): ?string;

  /**
   * Tests of the setting is set.
   *
   * @param string $config_name
   *   The name of the configuration/setting.
   * @param string $data_type
   *   The data type of the setting. E.g.: string, int, bool.
   *
   * @return bool
   *   TRUE if the setting exists, FALSE otherwise.
   */
  public function settingExists(string $config_name, string $data_type): bool;

  /**
   * Tests if the API is enabled.
   *
   * @param string $config_name
   *   The name of the configuration/setting.
   *
   * @return bool
   *   TRUE if the API is enabled, FALSE otherwise.
   */
  public function isEnabled(string $config_name): bool;

  /**
   * Tests if the IP address is an IPv4 address.
   *
   * @param mixed $ip_address
   *   The IP address to test.
   *
   * @return bool
   *   TRUE if the IP address is an IPv4 address, FALSE otherwise.
   */
  public function isIpv4(mixed $ip_address): bool;

}
