<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides a service for interacting with the ipinfo.io API.
 */
class IpInfoIo implements IpInfoIoInterface {

  use StringTranslationTrait;


  /**
   * The base URL for the ipinfo.io API.
   */
  protected const API_BASE_URL = 'https://ipinfo.io/';

  /**
   * The IP Info base service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface
   */
  protected IpInfoInterface $ipInfoBase;

  /**
   * Constructs a new IpInfoIo service.
   *
   * @param \Drupal\ip_info\Services\IpInfoInterface $ip_info_base
   *   The IP Info base service.
   */
  public function __construct(IpInfoInterface $ip_info_base) {
    $this->ipInfoBase = $ip_info_base;
  }

  /**
   * {@inheritdoc}
   */
  public function getProcessedData(string $ip_address): array {
    $data = [];
    $token = $this->ipInfoBase->getApiKey('ipinfo_io_token');

    if (empty($token)) {
      return $data;
    }

    $url = self::API_BASE_URL . $ip_address . '?token=' . $token;

    $ip_information = $this->ipInfoBase->getIpInformation($ip_address, $url, 'ipinfo.io');

    if ($ip_information) {
      if (array_key_exists('country', $ip_information)) {
        $data[] = ['label' => $this->t('Country'), 'value' => $ip_information['country']];
      }
      if (array_key_exists('region', $ip_information)) {
        $data[] = ['label' => $this->t('Region'), 'value' => $ip_information['region']];
      }
      if (array_key_exists('city', $ip_information)) {
        $data[] = ['label' => $this->t('City'), 'value' => $ip_information['city']];
      }
      if (array_key_exists('org', $ip_information)) {
        $data[] = ['label' => $this->t('Organization'), 'value' => $ip_information['org']];
      }
      if (array_key_exists('hostname', $ip_information)) {
        $data[] = ['label' => $this->t('Hostname'), 'value' => $ip_information['hostname']];
      }
      if (array_key_exists('bogon', $ip_information)) {
        $data[] = ['label' => $this->t('This ip is'), 'value' => 'Bogon'];
      }
    }

    return $data;
  }

}
