<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides a service for interacting with the ipstack.com API.
 */
class IpStackCom implements IpStackComInterface {

  use StringTranslationTrait;

  /**
   * The base URL for the ipstack.com API.
   */
  protected const API_BASE_URL = 'https://api.ipstack.com/';

  /**
   * The IP Info base service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface
   */
  protected IpInfoInterface $ipInfoBase;

  /**
   * Constructs a new IpStackCom service.
   *
   * @param \Drupal\ip_info\Services\IpInfoInterface $ip_info_base
   *   The IP Info base service.
   */
  public function __construct(IpInfoInterface $ip_info_base) {
    $this->ipInfoBase = $ip_info_base;
  }

  /**
   * {@inheritdoc}
   */
  public function getProcessedData(string $ip_address): array {
    $data = [];
    $key = $this->ipInfoBase->getApiKey('ipstack_com_key');

    if (empty($key)) {
      return $data;
    }

    $url = self::API_BASE_URL . $ip_address . '?access_key=' . $key;

    $ip_information = $this->ipInfoBase->getIpInformation($ip_address, $url, 'ipgeolocation.io');

    if ($ip_information) {
      if (array_key_exists('city', $ip_information)) {
        $location_info = $ip_information['city'];
        if (array_key_exists('region_name', $ip_information)) {
          $location_info .= ', ' . $ip_information['region_name'];
        }
        if (array_key_exists('country_name', $ip_information)) {
          $location_info .= ', ' . $ip_information['country_name'];
        }
        if (array_key_exists('location', $ip_information)) {
          if (array_key_exists('country_flag_emoji', $ip_information['location'])) {
            $location_info .= ' ' . $ip_information['location']['country_flag_emoji'];
          }
          if (array_key_exists('is_eu', $ip_information['location']) && $ip_information['location']['is_eu'] === TRUE) {
            $location_info .= ' (EU)';
          }
        }
        $data[] = ['label' => $this->t('Location'), 'value' => $location_info];
      }
      if (array_key_exists('connection', $ip_information)) {
        $connection_info = '';
        if (array_key_exists('asn', $ip_information['connection'])) {
          $connection_info .= 'ASN: ' . $ip_information['connection']['asn'];
        }
        if (array_key_exists('isp', $ip_information['connection'])) {
          $connection_info .= ', ISP: ' . $ip_information['connection']['isp'];
        }
        if (array_key_exists('carrier', $ip_information['connection'])) {
          $connection_info .= ', Carrier: ' . $ip_information['connection']['carrier'];
        }
        if (array_key_exists('connection_type', $ip_information)) {
          $connection_info .= ', Connection type: ' . $ip_information['connection_type'];
        }
        if (array_key_exists('ip_routing_type', $ip_information)) {
          $connection_info .= ', IP routing type: ' . $ip_information['ip_routing_type'];
        }
        $data[] = ['label' => $this->t('Connection'), 'value' => $connection_info];
      }
      if (array_key_exists('security', $ip_information)) {
        $this_is = '';
        if (array_key_exists('is_proxy', $ip_information['security']) && $ip_information['security']['is_proxy'] === TRUE) {
          $this_is .= 'Proxy';
        }
        if (array_key_exists('is_tor', $ip_information['security']) && $ip_information['security']['is_tor'] === TRUE) {
          if (!empty($this_is)) {
            $this_is .= ', ';
          }
          $this_is .= 'Tor';
        }
        if (array_key_exists('hosting_facility', $ip_information['security']) && $ip_information['security']['hosting_facility'] === TRUE) {
          if (!empty($this_is)) {
            $this_is .= ', ';
          }
          $this_is .= 'Hosting facility';
        }
        if (array_key_exists('is_crawler', $ip_information['security']) && $ip_information['security']['is_crawler'] === TRUE) {
          if (!empty($this_is)) {
            $this_is .= ', ';
          }
          $this_is .= 'Crawler';
        }
        if (!empty($this_is)) {
          $data[] = ['label' => $this->t('This is'), 'value' => $this_is];
        }

        $security_info = '';
        if (array_key_exists('threat_level', $ip_information['security']) && !empty($ip_information['security']['threat_level'])) {
          $security_info .= 'Threat level: ' . $ip_information['security']['threat_level'];
        }
        if (array_key_exists('threat_types', $ip_information['security']) && !empty($ip_information['security']['threat_types'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'Threat types: ' . $ip_information['security']['threat_types'];
        }
        if (array_key_exists('proxy_level', $ip_information['security']) && !empty($ip_information['security']['proxy_level'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'Proxy level: ' . $ip_information['security']['proxy_level'];
        }
        if (array_key_exists('proxy_type', $ip_information['security']) && !empty($ip_information['security']['proxy_type'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'Proxy type: ' . $ip_information['security']['proxy_type'];
        }
        if (array_key_exists('proxy_last_detected', $ip_information['security']) && !empty($ip_information['security']['proxy_last_detected'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'Proxy last detected: ' . $ip_information['security']['proxy_last_detected'];
        }
        if (array_key_exists('crawler_name', $ip_information['security']) && !empty($ip_information['security']['crawler_name'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'Crawler name: ' . $ip_information['security']['crawler_name'];
        }
        if (array_key_exists('crawler_type', $ip_information['security']) && !empty($ip_information['security']['crawler_type'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'Crawler type: ' . $ip_information['security']['crawler_type'];
        }
        if (array_key_exists('vpn_service', $ip_information['security']) && !empty($ip_information['security']['vpn_service'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'VPN service: ' . $ip_information['security']['vpn_service'];
        }
        if (array_key_exists('anonymizer_status', $ip_information['security']) && !empty($ip_information['security']['anonymizer_status'])) {
          if (!empty($security_info)) {
            $security_info .= ', ';
          }
          $security_info .= 'Anonymizer status: ' . $ip_information['security']['anonymizer_status'];
        }
        if (!empty($security_info)) {
          $data[] = ['label' => $this->t('Security'), 'value' => $security_info];
        }
      }

    }

    return $data;
  }

}
