<?php

declare(strict_types=1);

namespace Drupal\ip_info\Services;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for managing webform submissions lists and statistics.
 */
class WebformSubmissionsList implements WebformSubmissionsListInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs a new WebformSubmissionsList service.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    DateFormatterInterface $date_formatter,
    AccountProxyInterface $current_user,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->dateFormatter = $date_formatter;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public function isWebformModuleAvailable(): bool {
    return class_exists('\Drupal\webform\WebformSubmissionListBuilder') &&
           class_exists('\Drupal\webform\Entity\WebformSubmission') &&
           $this->entityTypeManager->hasDefinition('webform_submission');
  }

  /**
   * {@inheritdoc}
   */
  public function hasViewPermission(): bool {
    return $this->currentUser->hasPermission('view any webform submission') ||
           $this->currentUser->hasPermission('administer webform');
  }

  /**
   * {@inheritdoc}
   */
  public function buildSubmissionsList(string $ip_address, int $limit = 20): array {
    if (!$this->isWebformModuleAvailable() || !$this->hasViewPermission()) {
      return [];
    }

    $submissions = $this->loadSubmissionsByIp($ip_address, $limit);

    if (empty($submissions)) {
      return [];
    }

    // Build table headers.
    $header = [
      $this->t('Serial'),
      $this->t('Webform'),
      $this->t('Created'),
      $this->t('Changed'),
      $this->t('User'),
      $this->t('Status'),
      $this->t('Operations'),
    ];

    // Build table rows.
    $rows = [];
    foreach ($submissions as $submission) {
      $rows[] = $this->buildSubmissionRow($submission, $ip_address);
    }

    $build = [];
    $build['webform_submissions_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#attributes' => [
        'id' => 'webform-submissions-by-ip',
        'class' => ['webform-results-table'],
      ],
      '#empty' => $this->t('No webform submissions found for this IP address.'),
      '#attached' => [
        'library' => ['ip_info/webform-integration'],
      ],
    ];

    // Add summary information.
    $total = $this->getTotalSubmissionCount($ip_address);
    if ($total > 0) {
      $build['webform_submissions_info'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->formatPlural(
          $total,
          'Found 1 webform submission from this IP address. Showing the most recent submission.',
          'Found @count webform submissions from this IP address. Showing the @limit most recent submissions.',
          ['@limit' => min($limit, $total)]
        ) . '</p>',
      ];
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildSubmissionStats(string $ip_address): array {
    if (!$this->isWebformModuleAvailable() || !$this->hasViewPermission()) {
      return [];
    }

    $total = $this->getTotalSubmissionCount($ip_address);

    if ($total == 0) {
      return [];
    }

    $draft_count = $this->getDraftSubmissionCount($ip_address);
    $date_range = $this->getSubmissionDateRange($ip_address);
    $unique_webform_count = $this->getUniqueWebformCount($ip_address);

    $stats_items = [];
    $stats_items[] = $this->t('<strong>Total submissions:</strong> @total', ['@total' => $total]);
    $stats_items[] = $this->t('<strong>Completed:</strong> @completed', ['@completed' => $total - $draft_count]);

    if ($draft_count > 0) {
      $stats_items[] = $this->t('<strong>Drafts:</strong> @drafts', ['@drafts' => $draft_count]);
    }

    if ($date_range) {
      $stats_items[] = $this->t('<strong>Date range:</strong> @range', ['@range' => $date_range]);
    }

    $stats_items[] = $this->t('<strong>Unique webforms:</strong> @count', ['@count' => $unique_webform_count]);

    return [
      '#type' => 'markup',
      '#markup' => '<div class="webform-submission-stats"><p>' . implode(' | ', $stats_items) . '</p></div>',
    ];
  }

  /**
   * Load webform submissions by IP address.
   *
   * @param string $ip_address
   *   The IP address to filter by.
   * @param int $limit
   *   The maximum number of submissions to load.
   *
   * @return \Drupal\webform\WebformSubmissionInterface[]
   *   Array of webform submissions.
   */
  protected function loadSubmissionsByIp(string $ip_address, int $limit = 20): array {
    $query = $this->entityTypeManager->getStorage('webform_submission')->getQuery()
      ->condition('remote_addr', $ip_address)
      ->sort('created', 'DESC')
      ->range(0, $limit)
      ->accessCheck(TRUE);

    $submission_ids = $query->execute();

    if (empty($submission_ids)) {
      return [];
    }

    return $this->entityTypeManager->getStorage('webform_submission')->loadMultiple($submission_ids);
  }

  /**
   * Build a table row for a webform submission.
   *
   * @param mixed $submission
   *   The webform submission entity.
   * @param string $ip_address
   *   The IP address for context.
   *
   * @return array
   *   Table row data.
   */
  protected function buildSubmissionRow($submission, string $ip_address): array {
    // Safely get webform and owner using method_exists checks.
    $webform = NULL;
    $owner = NULL;

    if (method_exists($submission, 'getWebform')) {
      $webform = call_user_func([$submission, 'getWebform']);
    }

    if (method_exists($submission, 'getOwner')) {
      $owner = call_user_func([$submission, 'getOwner']);
    }

    // Create operations links.
    $operations = [];
    if (method_exists($submission, 'access') && method_exists($submission, 'toUrl')) {
      if (call_user_func([$submission, 'access'], 'view')) {
        $operations[] = Link::fromTextAndUrl(
          $this->t('View'),
          call_user_func([$submission, 'toUrl'], 'canonical')
        )->toString();
      }
      if (call_user_func([$submission, 'access'], 'update')) {
        $operations[] = Link::fromTextAndUrl(
          $this->t('Edit'),
          call_user_func([$submission, 'toUrl'], 'edit-form')
        )->toString();
      }
    }

    $operations_cell = !empty($operations) ? implode(' | ', $operations) : $this->t('No access');

    // Build the status with more detail.
    $status = [];
    if (method_exists($submission, 'isDraft') && call_user_func([$submission, 'isDraft'])) {
      $status[] = $this->t('Draft');
    }
    else {
      $status[] = $this->t('Completed');
    }
    if (method_exists($submission, 'isSticky') && call_user_func([$submission, 'isSticky'])) {
      $status[] = $this->t('Starred');
    }
    if (method_exists($submission, 'isLocked') && call_user_func([$submission, 'isLocked'])) {
      $status[] = $this->t('Locked');
    }

    // Get serial and ID safely.
    $serial = method_exists($submission, 'serial') ? call_user_func([$submission, 'serial']) : NULL;
    $id = method_exists($submission, 'id') ? call_user_func([$submission, 'id']) : '';

    // Get timestamps safely.
    $created_time = '';
    $changed_time = '';
    if (method_exists($submission, 'getCreatedTime')) {
      $created_time = $this->dateFormatter->format(call_user_func([$submission, 'getCreatedTime']), 'short');
    }
    if (method_exists($submission, 'getChangedTime')) {
      $changed_time = $this->dateFormatter->format(call_user_func([$submission, 'getChangedTime']), 'short');
    }

    // Get owner display safely.
    $owner_display = $this->t('Anonymous (@ip)', ['@ip' => $ip_address]);
    if ($owner && method_exists($owner, 'isAuthenticated') && call_user_func([$owner, 'isAuthenticated'])) {
      if (method_exists($owner, 'toLink')) {
        $owner_display = call_user_func([$owner, 'toLink'])->toString();
      }
    }

    $is_draft = method_exists($submission, 'isDraft') && call_user_func([$submission, 'isDraft']);

    return [
      'data' => [
        $serial ?: $id,
        $webform && method_exists($webform, 'toLink') ? call_user_func([$webform, 'toLink'])->toString() : $this->t('N/A'),
        $created_time,
        $changed_time,
        $owner_display,
        implode(', ', $status),
        ['data' => ['#markup' => $operations_cell]],
      ],
      'class' => $is_draft ? ['webform-submission-draft'] : [],
    ];
  }

  /**
   * Get total submission count for an IP address.
   *
   * @param string $ip_address
   *   The IP address to count submissions for.
   *
   * @return int
   *   The total number of submissions.
   */
  protected function getTotalSubmissionCount(string $ip_address): int {
    $query = $this->entityTypeManager->getStorage('webform_submission')->getQuery()
      ->condition('remote_addr', $ip_address)
      ->accessCheck(TRUE)
      ->count();

    return (int) $query->execute();
  }

  /**
   * Get draft submission count for an IP address.
   *
   * @param string $ip_address
   *   The IP address to count draft submissions for.
   *
   * @return int
   *   The number of draft submissions.
   */
  protected function getDraftSubmissionCount(string $ip_address): int {
    $query = $this->entityTypeManager->getStorage('webform_submission')->getQuery()
      ->condition('remote_addr', $ip_address)
      ->condition('in_draft', 1)
      ->accessCheck(TRUE)
      ->count();

    return (int) $query->execute();
  }

  /**
   * Get the date range of submissions for an IP address.
   *
   * @param string $ip_address
   *   The IP address to get date range for.
   *
   * @return string
   *   Formatted date range string.
   */
  protected function getSubmissionDateRange(string $ip_address): string {
    if (!$this->isWebformModuleAvailable()) {
      return '';
    }

    $base_query = $this->entityTypeManager->getStorage('webform_submission')->getQuery()
      ->condition('remote_addr', $ip_address)
      ->accessCheck(TRUE);

    // Get oldest submission.
    $oldest_query = clone $base_query;
    $oldest_id = $oldest_query
      ->sort('created', 'ASC')
      ->range(0, 1)
      ->execute();

    // Get newest submission.
    $newest_query = clone $base_query;
    $newest_id = $newest_query
      ->sort('created', 'DESC')
      ->range(0, 1)
      ->execute();

    if (empty($oldest_id) || empty($newest_id)) {
      return '';
    }

    try {
      $storage = $this->entityTypeManager->getStorage('webform_submission');
      $oldest = $storage->load(reset($oldest_id));
      $newest = $storage->load(reset($newest_id));

      if (!$oldest || !$newest) {
        return '';
      }

      // Use method_exists to check for webform-specific methods.
      if (method_exists($oldest, 'getCreatedTime') && method_exists($newest, 'getCreatedTime')) {
        $oldest_date = $this->dateFormatter->format(call_user_func([$oldest, 'getCreatedTime']), 'short');
        $newest_date = $this->dateFormatter->format(call_user_func([$newest, 'getCreatedTime']), 'short');
      }
      else {
        // Fallback to generic entity field access.
        $oldest_date = $this->dateFormatter->format($oldest->created->value, 'short');
        $newest_date = $this->dateFormatter->format($newest->created->value, 'short');
      }

      if ($oldest->id() == $newest->id()) {
        return $newest_date;
      }

      return $oldest_date . ' - ' . $newest_date;
    }
    catch (\Exception $e) {
      // Return empty string if there's any error accessing webform entities.
      return '';
    }
  }

  /**
   * Get the number of unique webforms used by submissions from an IP address.
   *
   * @param string $ip_address
   *   The IP address to count unique webforms for.
   *
   * @return int
   *   The number of unique webforms.
   */
  protected function getUniqueWebformCount(string $ip_address): int {
    if (!$this->isWebformModuleAvailable()) {
      return 0;
    }

    $query = $this->entityTypeManager->getStorage('webform_submission')->getQuery()
      ->condition('remote_addr', $ip_address)
      ->accessCheck(TRUE);

    $submission_ids = $query->execute();

    if (empty($submission_ids)) {
      return 0;
    }

    try {
      $submissions = $this->entityTypeManager->getStorage('webform_submission')->loadMultiple($submission_ids);
      $unique_webforms = [];

      foreach ($submissions as $submission) {
        // Try webform-specific method first.
        if (method_exists($submission, 'getWebform')) {
          $webform = call_user_func([$submission, 'getWebform']);
          if ($webform) {
            $unique_webforms[$webform->id()] = TRUE;
          }
        }
        elseif (method_exists($submission, 'get')) {
          // Fallback to generic entity field access.
          $webform_field = call_user_func([$submission, 'get'], 'webform');
          if ($webform_field && !$webform_field->isEmpty()) {
            $unique_webforms[$webform_field->target_id] = TRUE;
          }
        }
      }

      return count($unique_webforms);
    }
    catch (\Exception $e) {
      return 0;
    }
  }

}
