<?php

namespace Drupal\Tests\ip_info\Unit\Services;

use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\ip_info\Services\IpApiCo;
use Drupal\ip_info\Services\IpInfoInterface;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @coversDefaultClass \Drupal\ip_info\Services\IpApiCo
 * @group ip_info
 */
class IpApiCoTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The IpInfoBase service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $ipInfoBase;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $stringTranslation;

  /**
   * The service under test.
   *
   * @var \Drupal\ip_info\Services\IpApiCo
   */
  protected $ipApiCo;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->ipInfoBase = $this->prophesize(IpInfoInterface::class);
    $this->stringTranslation = $this->prophesize(TranslationInterface::class);

    $this->stringTranslation->translate(Argument::type('string'), [], [])
      ->will(function ($args) {
        return $args[0];
      });

    $this->ipApiCo = new IpApiCo($this->ipInfoBase->reveal());
    $this->ipApiCo->setStringTranslation($this->stringTranslation->reveal());
  }

  /**
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $this->assertInstanceOf(IpApiCo::class, $this->ipApiCo);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataDisabled(): void {
    $this->ipInfoBase->isEnabled('ipapi_co_enable')->willReturn(FALSE);
    $result = $this->ipApiCo->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoIpInformation(): void {
    $this->ipInfoBase->isEnabled('ipapi_co_enable')->willReturn(TRUE);
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipapi.co/8.8.8.8/json', 'ipapi.co')->willReturn(NULL);
    $result = $this->ipApiCo->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataLocation(): void {
    $this->ipInfoBase->isEnabled('ipapi_co_enable')->willReturn(TRUE);
    $ip_information = [
      'city' => 'Mountain View',
      'region' => 'California',
      'country_name' => 'United States',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipapi.co/8.8.8.8/json', 'ipapi.co')->willReturn($ip_information);
    $result = $this->ipApiCo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Mountain View, California, United States', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataLocationEu(): void {
    $this->ipInfoBase->isEnabled('ipapi_co_enable')->willReturn(TRUE);
    $ip_information = [
      'city' => 'Berlin',
      'region' => 'Berlin',
      'country_name' => 'Germany',
      'in_eu' => TRUE,
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipapi.co/8.8.8.8/json', 'ipapi.co')->willReturn($ip_information);
    $result = $this->ipApiCo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Berlin, Berlin, Germany (EU)', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataOrganization(): void {
    $this->ipInfoBase->isEnabled('ipapi_co_enable')->willReturn(TRUE);
    $ip_information = [
      'org' => 'Test Organization',
      'asn' => '12345',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipapi.co/8.8.8.8/json', 'ipapi.co')->willReturn($ip_information);
    $result = $this->ipApiCo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('12345 Test Organization', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataOrganizationNoAsn(): void {
    $this->ipInfoBase->isEnabled('ipapi_co_enable')->willReturn(TRUE);
    $ip_information = [
      'org' => 'Test Organization',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipapi.co/8.8.8.8/json', 'ipapi.co')->willReturn($ip_information);
    $result = $this->ipApiCo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Test Organization', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataAll(): void {
    $this->ipInfoBase->isEnabled('ipapi_co_enable')->willReturn(TRUE);
    $ip_information = [
      'city' => 'Mountain View',
      'region' => 'California',
      'country_name' => 'United States',
      'org' => 'Test Organization',
      'asn' => '12345',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipapi.co/8.8.8.8/json', 'ipapi.co')->willReturn($ip_information);
    $result = $this->ipApiCo->getProcessedData('8.8.8.8');
    $this->assertCount(2, $result);
    $this->assertEquals('Mountain View, California, United States', $result[0]['value']);
    $this->assertEquals('12345 Test Organization', $result[1]['value']);
  }

}
