<?php

namespace Drupal\Tests\ip_info\Unit\Services;

use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\ip_info\Services\IpApiIs;
use Drupal\ip_info\Services\IpInfoInterface;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @coversDefaultClass \Drupal\ip_info\Services\IpApiIs
 * @group ip_info
 */
class IpApiIsTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The IpInfoBase service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $ipInfoBase;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $stringTranslation;

  /**
   * The service under test.
   *
   * @var \Drupal\ip_info\Services\IpApiIs
   */
  protected $ipApiIs;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->ipInfoBase = $this->prophesize(IpInfoInterface::class);
    $this->stringTranslation = $this->prophesize(TranslationInterface::class);

    $this->stringTranslation->translate(Argument::type('string'), Argument::any(), Argument::any())
      ->will(function ($args) {
        // Return the untranslated string for simplicity.
        return $args[0];
      });

    $this->ipApiIs = new IpApiIs($this->ipInfoBase->reveal());
    $this->ipApiIs->setStringTranslation($this->stringTranslation->reveal());
  }

  /**
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $this->assertInstanceOf(IpApiIs::class, $this->ipApiIs);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoApiKey(): void {
    $this->ipInfoBase->getApiKey('ipapi_is_key')->willReturn('');
    $result = $this->ipApiIs->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoIpInformation(): void {
    $this->ipInfoBase->getApiKey('ipapi_is_key')->willReturn('test_key');
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipapi.is/?q=8.8.8.8&key=test_key', 'ipapi.is')->willReturn(NULL);
    $result = $this->ipApiIs->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataLocation(): void {
    $this->ipInfoBase->getApiKey('ipapi_is_key')->willReturn('test_key');
    $ip_information = [
      'location' => [
        'city' => 'Mountain View',
        'zip' => '94043',
        'state' => 'California',
        'country' => 'United States',
      ],
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipapi.is/?q=8.8.8.8&key=test_key', 'ipapi.is')->willReturn($ip_information);
    $result = $this->ipApiIs->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Mountain View (94043), California, United States', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataCompany(): void {
    $this->ipInfoBase->getApiKey('ipapi_is_key')->willReturn('test_key');
    $ip_information = [
      'company' => [
        'name' => 'Test Company',
        'type' => 'Private',
        'domain' => 'test.com',
        'abuser_score' => 0.8,
      ],
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipapi.is/?q=8.8.8.8&key=test_key', 'ipapi.is')->willReturn($ip_information);
    $result = $this->ipApiIs->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Test Company - Private (test.com); abuser score: 0.8', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataDatacenter(): void {
    $this->ipInfoBase->getApiKey('ipapi_is_key')->willReturn('test_key');
    $ip_information = [
      'datacenter' => [
        'datacenter' => 'Google Cloud Platform',
        'domain' => 'google.com',
      ],
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipapi.is/?q=8.8.8.8&key=test_key', 'ipapi.is')->willReturn($ip_information);
    $result = $this->ipApiIs->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Google Cloud Platform (google.com)', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataIpProperties(): void {
    $this->ipInfoBase->getApiKey('ipapi_is_key')->willReturn('test_key');
    $ip_information = [
      'is_bogon' => TRUE,
      'is_mobile' => TRUE,
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipapi.is/?q=8.8.8.8&key=test_key', 'ipapi.is')->willReturn($ip_information);
    $result = $this->ipApiIs->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Bogon, Mobile', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataAll(): void {
    $this->ipInfoBase->getApiKey('ipapi_is_key')->willReturn('test_key');
    $ip_information = [
      'location' => [
        'city' => 'Mountain View',
        'zip' => '94043',
        'state' => 'California',
        'country' => 'United States',
      ],
      'company' => [
        'name' => 'Test Company',
        'type' => 'Private',
        'domain' => 'test.com',
        'abuser_score' => 0.8,
      ],
      'datacenter' => [
        'datacenter' => 'Google Cloud Platform',
        'domain' => 'google.com',
      ],
      'is_bogon' => TRUE,
      'is_mobile' => TRUE,
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipapi.is/?q=8.8.8.8&key=test_key', 'ipapi.is')->willReturn($ip_information);
    $result = $this->ipApiIs->getProcessedData('8.8.8.8');
    $this->assertCount(4, $result);
    $this->assertEquals('Mountain View (94043), California, United States', $result[0]['value']);
    $this->assertEquals('Test Company - Private (test.com); abuser score: 0.8', $result[1]['value']);
    $this->assertEquals('Google Cloud Platform (google.com)', $result[2]['value']);
    $this->assertEquals('Bogon, Mobile', $result[3]['value']);
  }

}
