<?php

namespace Drupal\Tests\ip_info\Unit\Services;

use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\ip_info\Services\IpGeolocationIo;
use Drupal\ip_info\Services\IpInfoInterface;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @coversDefaultClass \Drupal\ip_info\Services\IpGeolocationIo
 * @group ip_info
 */
class IpGeolocationIoTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The IpInfoBase service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $ipInfoBase;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $stringTranslation;

  /**
   * The service under test.
   *
   * @var \Drupal\ip_info\Services\IpGeolocationIo
   */
  protected $ipGeolocationIo;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->ipInfoBase = $this->prophesize(IpInfoInterface::class);
    $this->stringTranslation = $this->prophesize(TranslationInterface::class);

    $this->stringTranslation->translate(Argument::type('string'), [], [])
      ->will(function ($args) {
        // Return the untranslated string for simplicity.
        return $args[0];
      });

    $this->ipGeolocationIo = new IpGeolocationIo($this->ipInfoBase->reveal());
    $this->ipGeolocationIo->setStringTranslation($this->stringTranslation->reveal());
  }

  /**
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $this->assertInstanceOf(IpGeolocationIo::class, $this->ipGeolocationIo);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoApiKey(): void {
    $this->ipInfoBase->getApiKey('ipgeolocation_io_key')->willReturn('');
    $result = $this->ipGeolocationIo->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoIpInformation(): void {
    $this->ipInfoBase->getApiKey('ipgeolocation_io_key')->willReturn('test_key');
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipgeolocation.io/ipgeo?apiKey=test_key&ip=8.8.8.8', 'ipgeolocation.io')->willReturn(NULL);
    $result = $this->ipGeolocationIo->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataLocation(): void {
    $this->ipInfoBase->getApiKey('ipgeolocation_io_key')->willReturn('test_key');
    $ip_information = [
      'city' => 'Mountain View',
      'state_prov' => 'California',
      'country_name' => 'United States',
      'country_emoji' => '🇺🇸',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipgeolocation.io/ipgeo?apiKey=test_key&ip=8.8.8.8', 'ipgeolocation.io')->willReturn($ip_information);
    $result = $this->ipGeolocationIo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Mountain View, California, United States 🇺🇸', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataLocationEu(): void {
    $this->ipInfoBase->getApiKey('ipgeolocation_io_key')->willReturn('test_key');
    $ip_information = [
      'city' => 'Berlin',
      'state_prov' => 'Berlin',
      'country_name' => 'Germany',
      'is_eu' => TRUE,
      'country_emoji' => '🇩🇪',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipgeolocation.io/ipgeo?apiKey=test_key&ip=8.8.8.8', 'ipgeolocation.io')->willReturn($ip_information);
    $result = $this->ipGeolocationIo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Berlin, Berlin, Germany 🇩🇪 (EU)', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataOrganization(): void {
    $this->ipInfoBase->getApiKey('ipgeolocation_io_key')->willReturn('test_key');
    $ip_information = [
      'organization' => 'Test Organization',
      'asn' => '12345',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipgeolocation.io/ipgeo?apiKey=test_key&ip=8.8.8.8', 'ipgeolocation.io')->willReturn($ip_information);
    $result = $this->ipGeolocationIo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('12345 Test Organization', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataIsp(): void {
    $this->ipInfoBase->getApiKey('ipgeolocation_io_key')->willReturn('test_key');
    $ip_information = [
      'isp' => 'Test ISP',
      'connection_type' => 'DSL',
      'hostname' => 'test.example.com',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipgeolocation.io/ipgeo?apiKey=test_key&ip=8.8.8.8', 'ipgeolocation.io')->willReturn($ip_information);
    $result = $this->ipGeolocationIo->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Test ISP, connection type: DSL, hostname: test.example.com', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataAll(): void {
    $this->ipInfoBase->getApiKey('ipgeolocation_io_key')->willReturn('test_key');
    $ip_information = [
      'city' => 'Mountain View',
      'state_prov' => 'California',
      'country_name' => 'United States',
      'country_emoji' => '🇺🇸',
      'organization' => 'Test Organization',
      'asn' => '12345',
      'isp' => 'Test ISP',
      'connection_type' => 'DSL',
      'hostname' => 'test.example.com',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipgeolocation.io/ipgeo?apiKey=test_key&ip=8.8.8.8', 'ipgeolocation.io')->willReturn($ip_information);
    $result = $this->ipGeolocationIo->getProcessedData('8.8.8.8');
    $this->assertCount(3, $result);
    $this->assertEquals('Mountain View, California, United States 🇺🇸', $result[0]['value']);
    $this->assertEquals('12345 Test Organization', $result[1]['value']);
    $this->assertEquals('Test ISP, connection type: DSL, hostname: test.example.com', $result[2]['value']);
  }

}
