<?php

namespace Drupal\Tests\ip_info\Unit\Services;

use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\ip_info\Services\IpInfoInterface;
use Drupal\ip_info\Services\IpInfoIo;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @coversDefaultClass \Drupal\ip_info\Services\IpInfoIo
 * @group ip_info
 */
class IpInfoIoTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The IpInfoBase service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $ipInfoBase;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $stringTranslation;

  /**
   * The service under test.
   *
   * @var \Drupal\ip_info\Services\IpInfoIo
   */
  protected $ipInfoIo;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->ipInfoBase = $this->prophesize(IpInfoInterface::class);
    $this->stringTranslation = $this->prophesize(TranslationInterface::class);

    // Simple translation mocking -  replace with more robust mocking if needed.
    $this->stringTranslation->translate(Argument::type('string'), [], [])
      ->will(function ($args) {
        return $args[0];
      });

    $this->ipInfoIo = new IpInfoIo($this->ipInfoBase->reveal());
    $this->ipInfoIo->setStringTranslation($this->stringTranslation->reveal());
  }

  /**
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $this->assertInstanceOf(IpInfoIo::class, $this->ipInfoIo);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoToken(): void {
    $this->ipInfoBase->getApiKey('ipinfo_io_token')->willReturn('');
    $result = $this->ipInfoIo->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoIpInformation(): void {
    $this->ipInfoBase->getApiKey('ipinfo_io_token')->willReturn('test_token');
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipinfo.io/8.8.8.8?token=test_token', 'ipinfo.io')->willReturn(NULL);
    $result = $this->ipInfoIo->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataWithInformation(): void {
    $this->ipInfoBase->getApiKey('ipinfo_io_token')->willReturn('test_token');
    $ip_information = [
      'country' => 'US',
      'region' => 'CA',
      'city' => 'Mountain View',
      'org' => 'Google',
      'hostname' => 'google.com',
      'bogon' => TRUE,
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://ipinfo.io/8.8.8.8?token=test_token', 'ipinfo.io')->willReturn($ip_information);
    $result = $this->ipInfoIo->getProcessedData('8.8.8.8');
    $this->assertCount(6, $result);
    $this->assertEquals('US', $result[0]['value']);
    $this->assertEquals('CA', $result[1]['value']);
    $this->assertEquals('Mountain View', $result[2]['value']);
    $this->assertEquals('Google', $result[3]['value']);
    $this->assertEquals('google.com', $result[4]['value']);
    $this->assertEquals('Bogon', $result[5]['value']);
  }

}
