<?php

namespace Drupal\Tests\ip_info\Unit\Services;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Site\Settings;
use Drupal\ip_info\Services\IpInfo;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @coversDefaultClass \Drupal\ip_info\Services\IpInfo
 * @group ip_info
 */
class IpInfoTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $keyRepository;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $loggerFactory;

  /**
   * The IpInfo service.
   *
   * @var \Drupal\ip_info\Services\IpInfo
   */
  protected $ipInfo;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->keyRepository = $this->prophesize(KeyRepositoryInterface::class);
    $this->configFactory = $this->prophesize(ConfigFactoryInterface::class);
    $this->loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);

    $mock = new MockHandler();
    $handler = HandlerStack::create($mock);
    $client = new Client(['handler' => $handler]);

    $this->ipInfo = new IpInfo($client, $this->keyRepository->reveal(), $this->configFactory->reveal(), $this->loggerFactory->reveal());
  }

  /**
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $this->assertInstanceOf(IpInfo::class, $this->ipInfo);
  }

  /**
   * @covers ::getIpInformation
   */
  public function testGetIpInformationSuccess(): void {
    $mock = new MockHandler([
      new Response(200, [], '{"country": "US", "city": "New York"}'),
    ]);
    $handler = HandlerStack::create($mock);
    $client = new Client(['handler' => $handler]);
    $ipInfo = new IpInfo($client, $this->keyRepository->reveal(), $this->configFactory->reveal(), $this->loggerFactory->reveal());

    $result = $ipInfo->getIpInformation('8.8.8.8', 'https://example.com/ip', 'test_api');
    $this->assertEquals(['country' => 'US', 'city' => 'New York'], $result);
  }

  /**
   * @covers ::isEnabled
   */
  public function testIsEnabledFromConfig(): void {
    $config = $this->prophesize(Config::class);
    $config->get('ipapi_co_enable')->willReturn(FALSE);
    $this->configFactory->get('ip_info.settings')->willReturn($config->reveal());

    $ipInfo = new IpInfo(new Client(), $this->keyRepository->reveal(), $this->configFactory->reveal(), $this->loggerFactory->reveal());
    $this->assertFalse($ipInfo->isEnabled('ipapi_co_enable'));
  }

  /**
   * @covers ::settingExists
   */
  public function testSettingExists(): void {
    // Create a new IpInfo instance.
    $ipInfo = new IpInfo(new Client(), $this->keyRepository->reveal(), $this->configFactory->reveal(), $this->loggerFactory->reveal());

    // Get the original settings instance.
    $settingsReflection = new \ReflectionClass(Settings::class);
    $settingsInstanceProperty = $settingsReflection->getProperty('instance');
    $settingsInstanceProperty->setAccessible(TRUE);
    $originalInstance = $settingsInstanceProperty->getValue();

    try {
      // Test with existing non-empty setting.
      $settings = new Settings([
        'ip_info' => [
          'string_setting' => 'value',
          'empty_string_setting' => '',
          'null_setting' => NULL,
          'true_setting' => TRUE,
          'false_setting' => FALSE,
          'zero_setting' => 0,
          'one_setting' => 1,
          'array_setting' => ['value'],
          'empty_array_setting' => [],
        ],
      ]);
      $settingsInstanceProperty->setValue(NULL, $settings);

      // Test setting with data type string.
      $this->assertTrue($ipInfo->settingExists('string_setting'));
      $this->assertTrue($ipInfo->settingExists('empty_string_setting'));
      $this->assertFalse($ipInfo->settingExists('non_existing_setting'));
      $this->assertFalse($ipInfo->settingExists('null_setting'));
      $this->assertFalse($ipInfo->settingExists('true_setting'));
      $this->assertFalse($ipInfo->settingExists('false_setting'));
      $this->assertFalse($ipInfo->settingExists('zero_setting'));
      $this->assertFalse($ipInfo->settingExists('one_setting'));
      $this->assertFalse($ipInfo->settingExists('array_setting'));
      $this->assertFalse($ipInfo->settingExists('empty_array_setting'));

      // Test setting with data type bool.
      $this->assertFalse($ipInfo->settingExists('string_setting', 'bool'));
      $this->assertFalse($ipInfo->settingExists('empty_string_setting', 'bool'));
      $this->assertFalse($ipInfo->settingExists('non_existing_setting', 'bool'));
      $this->assertFalse($ipInfo->settingExists('null_setting', 'bool'));
      $this->assertTrue($ipInfo->settingExists('true_setting', 'bool'));
      $this->assertTrue($ipInfo->settingExists('false_setting', 'bool'));
      $this->assertFalse($ipInfo->settingExists('zero_setting', 'bool'));
      $this->assertFalse($ipInfo->settingExists('one_setting', 'bool'));
      $this->assertFalse($ipInfo->settingExists('array_setting', 'bool'));
      $this->assertFalse($ipInfo->settingExists('empty_array_setting', 'bool'));

      // Test setting with data type int.
      $this->assertFalse($ipInfo->settingExists('string_setting', 'int'));
      $this->assertFalse($ipInfo->settingExists('empty_string_setting', 'int'));
      $this->assertFalse($ipInfo->settingExists('non_existing_setting', 'int'));
      $this->assertFalse($ipInfo->settingExists('null_setting', 'int'));
      $this->assertFalse($ipInfo->settingExists('true_setting', 'int'));
      $this->assertFalse($ipInfo->settingExists('false_setting', 'int'));
      $this->assertTrue($ipInfo->settingExists('zero_setting', 'int'));
      $this->assertTrue($ipInfo->settingExists('one_setting', 'int'));
      $this->assertFalse($ipInfo->settingExists('array_setting', 'int'));
      $this->assertFalse($ipInfo->settingExists('empty_array_setting', 'int'));

      // Test setting with data type array.
      $this->assertFalse($ipInfo->settingExists('string_setting', 'array'));
      $this->assertFalse($ipInfo->settingExists('empty_string_setting', 'array'));
      $this->assertFalse($ipInfo->settingExists('non_existing_setting', 'array'));
      $this->assertFalse($ipInfo->settingExists('null_setting', 'array'));
      $this->assertFalse($ipInfo->settingExists('true_setting', 'array'));
      $this->assertFalse($ipInfo->settingExists('false_setting', 'array'));
      $this->assertFalse($ipInfo->settingExists('zero_setting', 'array'));
      $this->assertFalse($ipInfo->settingExists('one_setting', 'array'));
      $this->assertTrue($ipInfo->settingExists('array_setting', 'array'));
      $this->assertTrue($ipInfo->settingExists('empty_array_setting', 'array'));

      // Test setting with data type null.
      $this->assertFalse($ipInfo->settingExists('string_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('empty_string_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('non_existing_setting', 'null'));
      $this->assertTrue($ipInfo->settingExists('null_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('true_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('false_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('zero_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('one_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('array_setting', 'null'));
      $this->assertFalse($ipInfo->settingExists('empty_array_setting', 'null'));

      // Test with null settings.
      $settings = new Settings([]);
      $settingsInstanceProperty->setValue(NULL, $settings);
      $this->assertFalse($ipInfo->settingExists('any_setting'));
    }
    finally {
      // Restore the original settings instance.
      $settingsInstanceProperty->setValue(NULL, $originalInstance);
    }
  }

  /**
   * @covers ::isIpv4
   */
  public function testIsIpv4(): void {
    $this->assertTrue($this->ipInfo->isIpv4('192.168.1.1'));
    $this->assertFalse($this->ipInfo->isIpv4('.192.168.1.1'));
    $this->assertFalse($this->ipInfo->isIpv4('192.168.1.1.'));
    $this->assertFalse($this->ipInfo->isIpv4('invalid_ip'));
    $this->assertFalse($this->ipInfo->isIpv4(NULL));
    $this->assertFalse($this->ipInfo->isIpv4(123));
    $this->assertFalse($this->ipInfo->isIpv4([]));
    $this->assertFalse($this->ipInfo->isIpv4(FALSE));
    $this->assertFalse($this->ipInfo->isIpv4(TRUE));
    $this->assertFalse($this->ipInfo->isIpv4(0));
    $this->assertFalse($this->ipInfo->isIpv4(1));
    $this->assertFalse($this->ipInfo->isIpv4(0.0));
    $this->assertFalse($this->ipInfo->isIpv4('2001:0db8:85a3:0000:0000:8a2e:0370:7334'));
    $this->assertFalse($this->ipInfo->isIpv4(''));
  }

}
