<?php

namespace Drupal\Tests\ip_info\Unit\Services;

use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\ip_info\Services\IpInfoInterface;
use Drupal\ip_info\Services\IpStackCom;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @coversDefaultClass \Drupal\ip_info\Services\IpStackCom
 * @group ip_info
 */
class IpStackComTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The IpInfoBase service.
   *
   * @var \Drupal\ip_info\Services\IpInfoInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $ipInfoBase;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $stringTranslation;

  /**
   * The service under test.
   *
   * @var \Drupal\ip_info\Services\IpStackCom
   */
  protected $ipStackCom;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->ipInfoBase = $this->prophesize(IpInfoInterface::class);
    $this->stringTranslation = $this->prophesize(TranslationInterface::class);

    $this->stringTranslation->translate(Argument::type('string'), [], [])
      ->will(function ($args) {
        return $args[0];
      });

    $this->ipStackCom = new IpStackCom($this->ipInfoBase->reveal());
    $this->ipStackCom->setStringTranslation($this->stringTranslation->reveal());
  }

  /**
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $this->assertInstanceOf(IpStackCom::class, $this->ipStackCom);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoApiKey(): void {
    $this->ipInfoBase->getApiKey('ipstack_com_key')->willReturn('');

    $result = $this->ipStackCom->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataNoIpInformation(): void {
    $this->ipInfoBase->getApiKey('ipstack_com_key')->willReturn('test_key');
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipstack.com/8.8.8.8?access_key=test_key', 'ipgeolocation.io')->willReturn(NULL);

    $result = $this->ipStackCom->getProcessedData('8.8.8.8');
    $this->assertEmpty($result);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataLocation(): void {
    $this->ipInfoBase->getApiKey('ipstack_com_key')->willReturn('test_key');
    $ip_information = [
      'city' => 'Mountain View',
      'region_name' => 'California',
      'country_name' => 'United States',
      'location' => [
        'country_flag_emoji' => '🇺🇸',
        'is_eu' => FALSE,
      ],
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipstack.com/8.8.8.8?access_key=test_key', 'ipgeolocation.io')->willReturn($ip_information);

    $result = $this->ipStackCom->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Mountain View, California, United States 🇺🇸', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataLocationEu(): void {
    $this->ipInfoBase->getApiKey('ipstack_com_key')->willReturn('test_key');
    $ip_information = [
      'city' => 'Berlin',
      'region_name' => 'Berlin',
      'country_name' => 'Germany',
      'location' => [
        'country_flag_emoji' => '🇩🇪',
        'is_eu' => TRUE,
      ],
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipstack.com/8.8.8.8?access_key=test_key', 'ipgeolocation.io')->willReturn($ip_information);

    $result = $this->ipStackCom->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('Berlin, Berlin, Germany 🇩🇪 (EU)', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataConnection(): void {
    $this->ipInfoBase->getApiKey('ipstack_com_key')->willReturn('test_key');
    $ip_information = [
      'connection' => [
        'asn' => '12345',
        'isp' => 'Test ISP',
        'carrier' => 'Test Carrier',
      ],
      'connection_type' => 'broadband',
      'ip_routing_type' => 'test',
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipstack.com/8.8.8.8?access_key=test_key', 'ipgeolocation.io')->willReturn($ip_information);

    $result = $this->ipStackCom->getProcessedData('8.8.8.8');
    $this->assertCount(1, $result);
    $this->assertEquals('ASN: 12345, ISP: Test ISP, Carrier: Test Carrier, Connection type: broadband, IP routing type: test', $result[0]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataSecurity(): void {
    $this->ipInfoBase->getApiKey('ipstack_com_key')->willReturn('test_key');
    $ip_information = [
      'security' => [
        'is_proxy' => TRUE,
        'is_tor' => TRUE,
        'hosting_facility' => TRUE,
        'is_crawler' => TRUE,
        'threat_level' => 'high',
        'threat_types' => 'spam',
        'proxy_level' => 'test',
        'proxy_type' => 'test',
        'proxy_last_detected' => 'test',
        'crawler_name' => 'test',
        'crawler_type' => 'test',
        'vpn_service' => 'test',
        'anonymizer_status' => 'test',
      ],
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipstack.com/8.8.8.8?access_key=test_key', 'ipgeolocation.io')->willReturn($ip_information);

    $result = $this->ipStackCom->getProcessedData('8.8.8.8');
    $this->assertCount(2, $result);
    $this->assertEquals('Proxy, Tor, Hosting facility, Crawler', $result[0]['value']);
    $this->assertEquals('Threat level: high, Threat types: spam, Proxy level: test, Proxy type: test, Proxy last detected: test, Crawler name: test, Crawler type: test, VPN service: test, Anonymizer status: test', $result[1]['value']);
  }

  /**
   * @covers ::getProcessedData
   */
  public function testGetProcessedDataAll(): void {
    $this->ipInfoBase->getApiKey('ipstack_com_key')->willReturn('test_key');
    $ip_information = [
      'city' => 'Mountain View',
      'region_name' => 'California',
      'country_name' => 'United States',
      'location' => [
        'country_flag_emoji' => '🇺🇸',
        'is_eu' => FALSE,
      ],
      'connection' => [
        'asn' => '12345',
        'isp' => 'Test ISP',
        'carrier' => 'Test Carrier',
      ],
      'connection_type' => 'broadband',
      'ip_routing_type' => 'test',
      'security' => [
        'is_proxy' => TRUE,
        'is_tor' => TRUE,
        'hosting_facility' => TRUE,
        'is_crawler' => TRUE,
        'threat_level' => 'high',
        'threat_types' => 'spam',
        'proxy_level' => 'test',
        'proxy_type' => 'test',
        'proxy_last_detected' => 'test',
        'crawler_name' => 'test',
        'crawler_type' => 'test',
        'vpn_service' => 'test',
        'anonymizer_status' => 'test',
      ],
    ];
    $this->ipInfoBase->getIpInformation('8.8.8.8', 'https://api.ipstack.com/8.8.8.8?access_key=test_key', 'ipgeolocation.io')->willReturn($ip_information);

    $result = $this->ipStackCom->getProcessedData('8.8.8.8');
    $this->assertCount(4, $result);
    $this->assertEquals('Mountain View, California, United States 🇺🇸', $result[0]['value']);
    $this->assertEquals('ASN: 12345, ISP: Test ISP, Carrier: Test Carrier, Connection type: broadband, IP routing type: test', $result[1]['value']);
    $this->assertEquals('Proxy, Tor, Hosting facility, Crawler', $result[2]['value']);
    $this->assertEquals('Threat level: high, Threat types: spam, Proxy level: test, Proxy type: test, Proxy last detected: test, Crawler name: test, Crawler type: test, VPN service: test, Anonymizer status: test', $result[3]['value']);
  }

}
