<?php

declare(strict_types=1);

namespace Drupal\ip_limiter\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatter;
use Drupal\Core\Link;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for Ip limiter routes.
 */
final class IpLimiterController extends ControllerBase {

  /**
   * The controller constructor.
   */
  public function __construct(
    protected readonly Connection $connection,
    protected DateFormatter $dateFormatter,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('database'),
      $container->get('date.formatter'),
    );
  }

  /**
   * A table of all ip addresses that are currently banned.
   *
   * @return array
   *   A render array representing the table.
   */
  public function bannedIps(): array {
    $query = $this->connection->select('ip_limiter_ban', 'b');
    $query->fields('b', [
      'ip_address',
      'created',
      'updated',
      'multiplier',
      'ban_end',
    ]);
    $query->orderBy('ban_end', 'DESC');
    $result = $query->execute();

    $rows = [];
    foreach ($result as $record) {
      $rows[] = [
        $record->ip_address,
        $this->dateFormatter->format($record->created, 'short'),
        $this->dateFormatter->format($record->updated, 'short'),
        $record->multiplier,
        $this->dateFormatter->format($record->ban_end, 'short'),
        Link::createFromRoute('Unban', 'ip_limiter.unban', ['ip_address' => $record->ip_address]),
      ];
    }

    return [
      '#theme' => 'table',
      '#header' => [
        $this->t('IP Address'),
        $this->t('Created'),
        $this->t('Updated'),
        $this->t('Multiplier'),
        $this->t('Ban Ends'),
        $this->t('Operations'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No banned IPs found.'),
    ];
  }

}
