<?php

declare(strict_types=1);

namespace Drupal\ip_limiter\EventSubscriber;

use Drupal\Component\Datetime\Time;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * IP Limiter event subscriber.
 */
final class IpLimiterSubscriber implements EventSubscriberInterface {

  private const CONFIG_NAME = 'ip_limiter.settings';
  public const RESPONSE_TYPE_429 = 429;
  public const RESPONSE_TYPE_403 = 403;
  public const RESPONSE_TYPE_404 = 404;

  /**
   * The IP Limiter settings.
   */
  private ImmutableConfig $ipLimiterSettings;

  /**
   * The IP address in question.
   */
  private string $ipAddress;

  public function __construct(
    private ConfigFactoryInterface $configFactory,
    private Connection $connection,
    private Time $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => ['onRequest'],
    ];
  }

  /**
   * Handles the request event.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onRequest(RequestEvent $event): void {
    $this->ipAddress = $event->getRequest()->getClientIp() ?? '';
    if (empty($this->ipAddress)) {
      return;
    }

    $ipLimiterSettings = $this->getSettings();
    $blockedPaths = $ipLimiterSettings->get('blocked_paths') ?? '';
    $routePaths = $ipLimiterSettings->get('blocked_routes') ?? '';

    // If no paths or routes are blocked, there is no need to continue.
    if (empty($blockedPaths) && empty($routePaths)) {
      return;
    }

    $globalRestrict = $ipLimiterSettings->get('global_restrict');
    $isVisitRestricted = $this->isCurrentPathRestricted($ipLimiterSettings, $event->getRequest()->getPathInfo())
      || (
        $event->getRequest()->attributes->get('_route') !== NULL
        && $this->isCurrentRouteRestricted($ipLimiterSettings, $event->getRequest()->attributes->get('_route'))
      );
    // If the current route/path is not restricted and there is no setup to
    // globally denied access, there is no need to continue.
    if (!$globalRestrict && !$isVisitRestricted) {
      return;
    }

    // If the IP address is banned, deny access, without any further actions.
    if ($this->isBanned($globalRestrict, $isVisitRestricted)) {
      $this->denyAccess($event);
      return;
    }

    // Log the visit and check if the IP address should be banned.
    $this->logVisitAndCheckBan();

    // If the IP address is banned after logging the new entry, deny access.
    if ($this->isBanned(TRUE, TRUE)) {
      $this->denyAccess($event);
    }
  }

  /**
   * Checks if the IP address is banned.
   *
   * @param bool $globalRestrict
   *   Whether the IP address is globally restricted.
   * @param bool $isVisitRestricted
   *   Whether the current visit is restricted.
   *
   * @return bool
   *   TRUE if the IP address is banned, FALSE otherwise.
   */
  private function isBanned(bool $globalRestrict, bool $isVisitRestricted): bool {
    $banEntry = $this->getBanEntry();
    return $banEntry && $banEntry['ban_end'] > $this->time->getRequestTime() && ($globalRestrict || $isVisitRestricted);
  }

  /**
   * Retrieves the ban entry for the IP address.
   *
   * @return array|null
   *   The ban entry, if found, NULL otherwise.
   */
  private function getBanEntry(): ?array {
    $query = $this->connection->select('ip_limiter_ban', 'b')
      ->fields('b', ['multiplier', 'ban_end'])
      ->condition('b.ip_address', $this->ipAddress);
    return $query->execute()->fetchAssoc() ?: NULL;
  }

  /**
   * Checks if the current path is restricted.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $ipLimiterSettings
   *   The IP Limiter settings.
   * @param string $currentPath
   *   The current path.
   *
   * @return string|null
   *   The restricted path, if found, NULL otherwise.
   */
  private function isCurrentPathRestricted(ImmutableConfig $ipLimiterSettings, string $currentPath): ?string {
    $blockedPaths = $ipLimiterSettings->get('blocked_paths') ?? '';
    if (empty($blockedPaths)) {
      return NULL;
    }

    $currentPath = trim($currentPath, " \n\r\t\v\0/");
    $blockedPaths = array_map('trim', explode("\n", trim($blockedPaths)));
    return in_array($currentPath, $blockedPaths) ? $currentPath : NULL;
  }

  /**
   * Checks if the current route is restricted.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $ipLimiterSettings
   *   The IP Limiter settings.
   * @param string $routeName
   *   The current route name.
   *
   * @return string|null
   *   The restricted route, if found, NULL otherwise.
   */
  private function isCurrentRouteRestricted(ImmutableConfig $ipLimiterSettings, string $routeName): ?string {
    $routePaths = $ipLimiterSettings->get('blocked_routes') ?? '';
    if (empty($routePaths)) {
      return NULL;
    }

    $routePaths = array_map('trim', explode("\n", trim($routePaths)));
    return in_array($routeName, $routePaths) ? $routeName : NULL;
  }

  /**
   * Denies access to the request.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  private function denyAccess(RequestEvent $event): void {
    $responseType = $this->getSettings()->get('response_type') ?? self::RESPONSE_TYPE_403;
    $response = new Response('', $responseType);
    if ($responseType === self::RESPONSE_TYPE_403) {
      $response->setContent('Access denied');
    }
    elseif ($responseType === self::RESPONSE_TYPE_404) {
      $response->setContent('Page not found');
    }
    elseif ($responseType === self::RESPONSE_TYPE_429) {
      $response->setContent('Too many requests');
    }

    $event->setResponse($response);
  }

  /**
   * Logs the visit and checks if the IP address should be banned.
   */
  private function logVisitAndCheckBan(): void {
    $this->logVisit();

    $timeLimit = $this->time->getRequestTime() - $this->getSettings()->get('time_period');
    $count = $this->connection->select('ip_limiter_log', 'l')
      ->condition('l.ip_address', $this->ipAddress)
      ->condition('l.timestamp', $timeLimit, '>')
      ->countQuery()
      ->execute()
      ->fetchField();

    if ($count >= $this->getSettings()->get('max_requests')) {
      $this->banIpAddress();
    }
  }

  /**
   * Logs the visit.
   */
  private function logVisit(): void {
    $this->connection->insert('ip_limiter_log')
      ->fields([
        'ip_address' => $this->ipAddress,
        'timestamp' => $this->time->getRequestTime(),
      ])
      ->execute();
  }

  /**
   * Bans the IP address.
   */
  private function banIpAddress(): void {
    $banEntry = $this->getBanEntry();
    $banDuration = $this->getSettings()->get('ban_duration');
    $requestTime = $this->time->getRequestTime();
    if ($banEntry) {
      $newMultiplier = $banEntry['multiplier'] * 2;
      $this->connection->update('ip_limiter_ban')
        ->fields([
          'ip_address' => $this->ipAddress,
          'multiplier' => $newMultiplier,
          'ban_end' => $requestTime + ($banDuration * $newMultiplier),
        ])
        ->condition('ip_address', $this->ipAddress)
        ->execute();
    }
    else {
      $this->connection->insert('ip_limiter_ban')
        ->fields([
          'ip_address' => $this->ipAddress,
          'multiplier' => 1,
          'created' => $requestTime,
          'updated' => $requestTime,
          'ban_end' => $requestTime + $banDuration,
        ])
        ->execute();
    }

    $this->connection->delete('ip_limiter_log')
      ->condition('ip_address', $this->ipAddress)
      ->execute();
  }

  /**
   * Retrieves the IP Limiter settings.
   *
   * @return \Drupal\Core\Config\ImmutableConfig
   *   The IP Limiter settings.
   */
  private function getSettings(): ImmutableConfig {
    if (!isset($this->ipLimiterSettings)) {
      $this->ipLimiterSettings = $this->configFactory->get(self::CONFIG_NAME);
    }
    return $this->ipLimiterSettings;
  }

}
